package svc

import (
	"context"
	"fmt"
	"time"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"code.justin.tv/video/brassclient"
	jwt "github.com/dgrijalva/jwt-go"
	"github.com/twitchtv/twirp"
)

func (c *client) GetChannelPageInfo(ctx context.Context, request *parsnip.GetChannelPageInfoRequest) (*parsnip.GetChannelPageInfoResponse, error) {
	item, err := c.ddbGetChannelByID(request.ChannelId)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to retrieve channel from dynamodb")
		return nil, twirp.InternalError("Failed to retrieve channel")
	}

	channel, err := c.hydrateChannel(item)
	if err != nil {
		// TODO: Work out how to do better error handling depending on the type of error that occurs
		c.logger.WithError(err).Warnf("Failed to retrieve channel")
		return nil, twirp.InternalError("Failed to retrieve channel")
	}

	// Check the bindle lock via BRASS if it's specified on the channel
	if channel.BindleLockId != "" && c.brass != nil {
		userID := UserID(ctx)
		if userID == "" {
			return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
		}

		resp, err := c.brass.IsAuthorized(context.Background(), &brassclient.IsAuthorizedRequest{
			Operation: brassclient.OperationUnlock,
			Actor: &brassclient.ActorReference{
				Type: brassclient.ActorTypePrincipal,
				ID:   userID,
			},
			Resource: &brassclient.ResourceReference{
				Type:      brassclient.ResourceTypeLock,
				Namespace: brassclient.NamespaceBindle,
				Name:      channel.BindleLockId,
			},
		})

		if err != nil {
			c.logger.WithError(err).Warnf("BRASS IsAuthorized user=%q lock=%q", userID, channel.BindleLockId)
			return nil, twirp.InternalError("Failed to establish permissions for stream")
		}

		if !resp.Authorized {
			return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to view this stream")
		}
	}

	token := jwt.NewWithClaims(jwt.SigningMethodES384, jwt.MapClaims{
		"aws:channel-arn":                 channel.IvsChannelArn,
		"aws:access-control-allow-origin": c.acao,
		"exp":                             time.Now().Add(c.tokenExpiry).Unix(),
	})

	// Sign and get the complete encoded token as a string using the secret
	tokenString, err := token.SignedString(c.playbackKey)
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to generate playback token")
		return nil, twirp.InternalError("Failed to generate playback token")
	}

	return &parsnip.GetChannelPageInfoResponse{
		ChannelTitle:       channel.ChannelTitle,
		ChannelDescription: channel.ChannelDescription,
		PlaybackUrl:        channel.IvsPlaybackUrl,
		AuthToken:          tokenString,
	}, nil
}
