package svc

import (
	"context"
	"fmt"

	rpc "code.justin.tv/event-engineering/parsnip/pkg/rpc"

	"code.justin.tv/video/brassclient"
	"github.com/twitchtv/twirp"
)

func (c *client) GetModuleAccess(ctx context.Context, request *rpc.GetModuleAccessRequest) (*rpc.GetModuleAccessResponse, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)

	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	return &rpc.GetModuleAccessResponse{
		IsAdmin:     isAdmin,
		IsPublisher: isPublisher,
	}, nil
}

func (c *client) getPermissions(ctx context.Context) (isAdmin, isPublisher bool, err error) {
	// This happens when testing on dev so just allow it
	if c.brass == nil {
		isAdmin = true
		isPublisher = true
		return
	}

	userID := UserID(ctx)
	if userID == "" {
		return false, false, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// It's important that the order of these matches the order of the bools in the response because... that's how BRASS works
	resources := []*brassclient.ResourceReference{
		&brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      c.bindleLocks.IsAdminBindleLockID,
		},
		&brassclient.ResourceReference{
			Type:      brassclient.ResourceTypeLock,
			Namespace: brassclient.NamespaceBindle,
			Name:      c.bindleLocks.IsPublisherBindleLockID,
		},
	}

	resp, err := c.brass.BatchIsAuthorized(ctx, &brassclient.BatchIsAuthorizedRequest{
		Operation: brassclient.OperationUnlock,
		Actor: &brassclient.ActorReference{
			Type: brassclient.ActorTypePrincipal,
			ID:   userID,
		},
		Resources: resources,
	})

	if err != nil {
		return false, false, fmt.Errorf("BRASS BatchIsAuthorized user=%q %w", userID, err)
	}

	if len(resp.Authorizations) != len(resources) {
		return false, false, fmt.Errorf("Expected %v authorizations, got %v", len(resources), len(resp.Authorizations))
	}

	for _, authorisationResult := range resp.Authorizations {
		switch authorisationResult.Resource.Name {
		case c.bindleLocks.IsAdminBindleLockID:
			isAdmin = authorisationResult.Authorized
			break
		case c.bindleLocks.IsPublisherBindleLockID:
			isPublisher = authorisationResult.Authorized
			break
		}
	}

	return
}
