package svc

import (
	"context"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/google/uuid"
	"github.com/twitchtv/twirp"
)

type ddbQuestion struct {
	QuestionId    string               `protobuf:"bytes,1,opt,name=question_id,json=questionId,proto3" json:"question_id"`
	ChannelId     string               `protobuf:"bytes,2,opt,name=channel_id,json=channelId,proto3" json:"channel_id"`
	HashCreatedBy string               `protobuf:"bytes,3,opt,name=hash_created_by,json=hashCreatedBy,proto3" json:"hash_created_by"`
	CreatedOn     *timestamp.Timestamp `protobuf:"bytes,4,opt,name=created_on,json=createdOn,proto3" json:"created_on"`
	IsAnswered    bool                 `protobuf:"varint,5,opt,name=is_answered,json=isAnswered,proto3" json:"is_answered"`
	AnsweredOn    *timestamp.Timestamp `protobuf:"bytes,6,opt,name=answered_on,json=answeredOn,proto3" json:"answered_on"`
	Upvotes       int64                `protobuf:"varint,7,opt,name=upvotes,proto3" json:"upvotes"`
	Description   string               `protobuf:"bytes,8,opt,name=description,proto3" json:"description"`
	VotedMap      map[string]bool      `json:"voted_map,omitempty"`
}

func (c *client) CreateQuestion(ctx context.Context, request *parsnip.CreateQuestionRequest) (*parsnip.Question, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	_, err = c.GetChannel(ctx, &parsnip.GetChannelRequest{
		ChannelIdHash: getChannelIDHash(request.ChannelId),
	})

	if err != nil {
		return nil, err
	}

	questionIDByte, err := uuid.NewUUID()

	if err != nil {
		c.logger.WithError(err).Warn("Failed to generate question_id uuid")

		return nil, twirp.InternalError("Failed to create a question")
	}

	questionID := questionIDByte.String()
	createdOn := ptypes.TimestampNow()

	ddbq := &ddbQuestion{
		QuestionId:    questionID,
		ChannelId:     request.ChannelId,
		HashCreatedBy: request.HashCreatedBy,
		CreatedOn:     createdOn,
		Description:   request.Description,
	}

	item, err := dynamodbattribute.MarshalMap(ddbq)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal ddbQuestion")

		return nil, twirp.InternalError("Failed to create a question")
	}

	_, err = c.ddb.PutItem(&dynamodb.PutItemInput{
		TableName:           aws.String(c.questionsTableName),
		Item:                item,
		ConditionExpression: aws.String("attribute_not_exists(question_id)"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to write channel to dynamodb")

		return nil, twirp.InternalError("Failed to create a question")
	}

	return &parsnip.Question{
		QuestionId:    questionID,
		ChannelId:     request.ChannelId,
		HashCreatedBy: request.HashCreatedBy,
		CreatedOn:     createdOn,
		Description:   request.Description,
	}, nil
}

func (c *client) GetQuestion(ctx context.Context, request *parsnip.GetQuestionRequest) (*parsnip.Question, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	_, err = c.GetChannel(ctx, &parsnip.GetChannelRequest{
		ChannelIdHash: getChannelIDHash(request.ChannelId),
	})

	if err != nil {
		return nil, err
	}

	item, err := c.ddbGetQuestion(request.QuestionId)

	if err != nil {
		return nil, err
	}

	channel, err := hydrateQuestion(item)

	if err != nil {
		// TODO: Work out how to do better error handling depending on the type of error that occurs
		c.logger.WithError(err).Warnf("Failed to retrieve channel")
		return nil, twirp.InternalError("Failed to retrieve channel")
	}

	return channel, nil
}

func (c *client) ddbGetQuestion(questionIDHash string) (map[string]*dynamodb.AttributeValue, error) {
	idAttr, err := dynamodbattribute.Marshal(questionIDHash)
	if err != nil {
		return nil, err
	}

	resp, err := c.ddb.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(c.questionsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"question_id": idAttr,
		},
	})

	if err != nil {
		return nil, err
	}

	return resp.Item, nil
}

func hydrateQuestion(item map[string]*dynamodb.AttributeValue) (*parsnip.Question, error) {
	var ddbq ddbQuestion
	err := dynamodbattribute.ConvertFromMap(item, &ddbq)

	if err != nil {
		return nil, err
	}

	return convertDDBQuestion(&ddbq), nil
}

func convertDDBQuestion(ddbq *ddbQuestion) *parsnip.Question {
	return &parsnip.Question{
		QuestionId:    ddbq.QuestionId,
		ChannelId:     ddbq.ChannelId,
		HashCreatedBy: ddbq.HashCreatedBy,
		CreatedOn:     ddbq.CreatedOn,
		IsAnswered:    ddbq.IsAnswered,
		AnsweredOn:    ddbq.AnsweredOn,
		Upvotes:       ddbq.Upvotes,
		Description:   ddbq.Description,
	}
}

func (c *client) GetQuestions(ctx context.Context, request *parsnip.GetQuestionsRequest) (*parsnip.ListOf_Questions, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	channelIDAttr, err := dynamodbattribute.Marshal(request.ChannelId)
	if err != nil {
		return nil, err
	}

	resp, err := c.ddb.Scan(&dynamodb.ScanInput{
		TableName:        aws.String(c.questionsTableName),
		IndexName:        aws.String("idx_by_channel_id"),
		FilterExpression: aws.String("channel_id = :channel_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channel_id": channelIDAttr,
		},
	})

	if err != nil {
		return nil, err
	}

	var questions []*ddbQuestion

	err = dynamodbattribute.UnmarshalListOfMaps(resp.Items, &questions)

	if err != nil {
		return nil, err
	}

	result := &parsnip.ListOf_Questions{
		Questions: make([]*parsnip.Question, 0, len(questions)),
	}

	for _, qu := range questions {
		convertedModel := convertDDBQuestion(qu)
		result.Questions = append(result.Questions, convertedModel)
	}

	return result, nil
}

func (c *client) UpvoteQuestion(ctx context.Context, request *parsnip.UpvoteQuestionRequest) (*parsnip.Question, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	questionIDAttr, err := dynamodbattribute.Marshal(request.QuestionId)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal question ID")
	}

	updateExp := "ADD upvotes :inc_num"
	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.questionsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"question_id": questionIDAttr,
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":inc_num": &dynamodb.AttributeValue{N: aws.String("1")},
		},
		UpdateExpression: &updateExp,
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update question")
	}

	return c.GetQuestion(ctx, &parsnip.GetQuestionRequest{
		QuestionId: request.QuestionId,
		ChannelId:  request.ChannelId,
	})
}

func (c *client) DeleteAllQuestions(ctx context.Context, request *parsnip.DeleteAllQuestionsRequest) (*parsnip.DeleteAllQuestionsResponse, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin || !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	questions, err := c.GetQuestions(ctx, &parsnip.GetQuestionsRequest{
		ChannelId:  request.ChannelId,
		HashUserId: request.HashUserId,
	})
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to retrieve questions")
		return nil, err
	}

	for _, q := range questions.Questions {
		_, err := c.DeleteQuestion(ctx, &parsnip.DeleteQuestionRequest{
			QuestionId: q.QuestionId,
			ChannelId:  request.ChannelId,
			HashUserId: request.HashUserId,
		})

		if err != nil {
			return nil, err
		}
	}
	return &parsnip.DeleteAllQuestionsResponse{}, nil
}

func (c *client) DeleteQuestion(ctx context.Context, request *parsnip.DeleteQuestionRequest) (*parsnip.DeleteQuestionResponse, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin || !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	question, err := c.GetQuestion(ctx, &parsnip.GetQuestionRequest{
		QuestionId: request.QuestionId,
		ChannelId:  request.ChannelId,
		HashUserId: request.HashUserId,
	})
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to retrieve question")
		return nil, err
	}

	if !isPublisher && question.HashCreatedBy != request.HashUserId {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	idAttr, err := dynamodbattribute.Marshal(request.QuestionId)
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to generate ID attribute")
		return nil, twirp.InternalError("Failed to delete question")
	}

	_, err = c.ddb.DeleteItem(&dynamodb.DeleteItemInput{
		TableName: aws.String(c.questionsTableName),
		Key:       map[string]*dynamodb.AttributeValue{"question_id": idAttr},
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to delete channel from dynamodb")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	return &parsnip.DeleteQuestionResponse{}, nil
}

func (c *client) AnswerQuestion(ctx context.Context, request *parsnip.AnswerQuestionRequest) (*parsnip.Question, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin || !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	_, err = c.GetQuestion(ctx, &parsnip.GetQuestionRequest{
		ChannelId:  request.ChannelId,
		QuestionId: request.QuestionId,
	})
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to retrieve questions")
		return nil, err
	}

	questionIDAttr, err := dynamodbattribute.Marshal(request.QuestionId)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal question ID")
	}

	updateExp := "SET is_answered = :bool_true"
	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.questionsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"question_id": questionIDAttr,
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":bool_true": &dynamodb.AttributeValue{BOOL: aws.Bool(true)},
		},
		UpdateExpression: &updateExp,
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update question")
	}

	return c.GetQuestion(ctx, &parsnip.GetQuestionRequest{
		QuestionId: request.QuestionId,
		ChannelId:  request.ChannelId,
	})
}
