package svc

import (
	"context"
	"crypto/ecdsa"
	"crypto/sha256"
	"fmt"
	"time"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"code.justin.tv/video/brassclient"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/ivs/ivsiface"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/aws/aws-sdk-go/service/ssm/ssmiface"
	jwt "github.com/dgrijalva/jwt-go"
	"github.com/sirupsen/logrus"
)

// Client represents the RPCs for this service
type Client interface {
	GetModuleAccess(ctx context.Context, request *parsnip.GetModuleAccessRequest) (*parsnip.GetModuleAccessResponse, error)
	CreateChannel(context.Context, *parsnip.CreateChannelRequest) (*parsnip.ChannelInfo, error)
	UpdateChannel(context.Context, *parsnip.UpdateChannelRequest) (*parsnip.ChannelInfo, error)
	DeleteChannel(context.Context, *parsnip.DeleteChannelRequest) (*parsnip.DeleteChannelResponse, error)
	GetChannel(context.Context, *parsnip.GetChannelRequest) (*parsnip.ChannelInfo, error)
	ListChannels(ctx context.Context, request *parsnip.ListChannelsRequest) (*parsnip.ListChannelsResponse, error)
	GetChannelPageInfo(ctx context.Context, request *parsnip.GetChannelPageInfoRequest) (*parsnip.GetChannelPageInfoResponse, error)

	CreateQuestion(context.Context, *parsnip.CreateQuestionRequest) (*parsnip.Question, error)
	GetQuestion(context.Context, *parsnip.GetQuestionRequest) (*parsnip.Question, error)
	GetQuestions(context.Context, *parsnip.GetQuestionsRequest) (*parsnip.ListOf_Questions, error)
	UpvoteQuestion(context.Context, *parsnip.UpvoteQuestionRequest) (*parsnip.Question, error)
	DeleteQuestion(context.Context, *parsnip.DeleteQuestionRequest) (*parsnip.DeleteQuestionResponse, error)
	DeleteAllQuestions(context.Context, *parsnip.DeleteAllQuestionsRequest) (*parsnip.DeleteAllQuestionsResponse, error)
	AnswerQuestion(context.Context, *parsnip.AnswerQuestionRequest) (*parsnip.Question, error)
}

type client struct {
	channelsTableName  string
	questionsTableName string
	playbackKey        *ecdsa.PrivateKey
	ddb                dynamodbiface.DynamoDBAPI
	ivs                ivsiface.IVSAPI
	brass              *brassclient.BRASS
	bindleLocks        BindleLockConfig
	tokenExpiry        time.Duration
	acao               string
	logger             logrus.FieldLogger
}

// BindleLockConfig keeps all the bindle config together to make it a bit easier to manage
type BindleLockConfig struct {
	IsAdminBindleLockID     string
	IsPublisherBindleLockID string
}

// New creates the Parsnip service object
func New(channelsTableName, questionsTableName, playbackAuthKeyParameterName string, tokenExpiry time.Duration, acao string, ddb dynamodbiface.DynamoDBAPI, ivs ivsiface.IVSAPI, ssmClient ssmiface.SSMAPI, brass *brassclient.BRASS, bindleLocks BindleLockConfig, logger logrus.FieldLogger) (Client, error) {
	ssmResponse, err := ssmClient.GetParameter(&ssm.GetParameterInput{
		Name:           aws.String(playbackAuthKeyParameterName),
		WithDecryption: aws.Bool(true),
	})

	if err != nil {
		return nil, err
	}

	privateKey, err := jwt.ParseECPrivateKeyFromPEM([]byte(*ssmResponse.Parameter.Value))

	if err != nil {
		return nil, err
	}

	return &client{
		channelsTableName:  channelsTableName,
		questionsTableName: questionsTableName,
		ddb:                ddb,
		ivs:                ivs,
		brass:              brass,
		bindleLocks:        bindleLocks,
		tokenExpiry:        tokenExpiry,
		acao:               acao,
		playbackKey:        privateKey,
		logger:             logger,
	}, nil
}

func getChannelIDHash(channelID string) string {
	return fmt.Sprintf("%x", sha256.Sum256([]byte(channelID)))
}

type ddbChannel struct {
	ChannelIDHash      string `json:"channel_id_hash"`
	ChannelID          string `json:"channel_id"`
	BindleLockID       string `json:"bindle_lock_id"`
	IVSChannelARN      string `json:"ivs_channel_arn"`
	IVSStreamKeyARN    string `json:"ivs_stream_key_arn"`
	ChannelTitle       string `json:"channel_title"`
	ChannelDescription string `json:"channel_description"`
}
