package svc

import (
	"context"

	parsnip "code.justin.tv/event-engineering/parsnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/twitchtv/twirp"
)

func (c *client) UpdateChannel(ctx context.Context, request *parsnip.UpdateChannelRequest) (*parsnip.ChannelInfo, error) {
	isAdmin, isPublisher, err := c.getPermissions(ctx)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to establish permissions")
		return nil, err
	}

	if !isAdmin && !isPublisher {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	channelIDHashAttr, err := dynamodbattribute.Marshal(request.ChannelIdHash)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update endpoint status: Marshal Channel ID Hash")
	}

	// Create attributes for all the fields we're updating
	bindleLockAttr, err := dynamodbattribute.Marshal(request.BindleLockId)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel: Marshal Bindle Lock ID")
	}

	channelTitleAttr, err := dynamodbattribute.Marshal(request.ChannelTitle)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel: Marshal Channel Title")
	}

	channelDescriptionAttr, err := dynamodbattribute.Marshal(request.ChannelDescription)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel: Marshal Channel Description")
	}

	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.channelsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id_hash": channelIDHashAttr,
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":bindle_lock_id":      bindleLockAttr,
			":channel_id_hash":     channelIDHashAttr,
			":channel_title":       channelTitleAttr,
			":channel_description": channelDescriptionAttr,
		},
		UpdateExpression:    aws.String("SET bindle_lock_id=:bindle_lock_id, channel_title=:channel_title, channel_description=:channel_description"),
		ConditionExpression: aws.String("channel_id_hash=:channel_id_hash"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel")
	}

	return c.GetChannel(ctx, &parsnip.GetChannelRequest{
		ChannelIdHash: request.ChannelIdHash,
	})
}
