import React from "react";
import { BrowserRouter, Route, Switch } from "react-router-dom";
import {
  AnimatedGlitchLogo,
  CoreUIRoot,
  Display,
  FontSize,
  FlexDirection,
  StyledLayout,
  Layout,
} from "twitch-core-ui";
import { HomePage } from "./pages/home";
import {
  AdminHomePage,
  AdminChannelDetailPage,
  AdminNewChannelPage,
} from "./pages/admin";
import { ChannelPage } from "./pages/channel";
import { QuestionsPage } from "./pages/questions";
import { InfoBar } from "./components/info-bar";
import { API } from "./lib/api";
import { Theme } from "twitch-core-ui-tokens";

import "./App.css";
import "twitch-core-ui/css/index.css";

const kc = [38, 38, 40, 40, 37, 39, 37, 39, 66, 65];
const kcClass = "these-are-not-the-droids-youre-looking-for";

export interface State {
  userInfo: UserInfo | null;
  userThemeValue: String;
}

export class App extends React.Component<{}, State> {
  public state: State = {
    userInfo: null,
    userThemeValue: window.localStorage.getItem("userThemeValue"),
  };

  private kb: number[] = [];
  private appClass: string = "";

  public render() {
    var themeValue: Theme | "system";
    switch (this.state.userThemeValue) {
      case "light":
        themeValue = "light";
        break;
      case "dark":
        themeValue = "dark";
        break;
      case "system":
        themeValue = "system";
        break;
      default:
        // Set the default to follow the system theme
        window.localStorage.setItem("userThemeValue", "system");
        themeValue = "system";
        break;
    }
    return (
      <BrowserRouter>
        <CoreUIRoot theme={themeValue} />
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          fullHeight
        >
          <Layout flexGrow={0}>
            <InfoBar
              userInfo={this.state.userInfo}
              updateThemeState={this.updateThemeState.bind(this)}
              currentTheme={window.localStorage.getItem("userThemeValue")}
            />
          </Layout>
          <Layout flexGrow={1}>
            <Switch>
              <Route exact path="/" component={HomePage} />
              <Route exact path="/admin" component={AdminHomePage} />
              <Route
                exact
                path="/admin/channel/new"
                component={AdminNewChannelPage}
              />
              <Route
                exact
                path="/admin/channel/:channel_id_hash"
                children={({ match }) => (
                  <AdminChannelDetailPage
                    match={match}
                    userInfo={this.state.userInfo}
                  />
                )}
              />
              <Route
                path="/:channel_id/questions"
                component={QuestionsPage}
                userInfo={this.state.userInfo}
              />
              <Route
                path="/:channel_id"
                component={ChannelPage}
                userInfo={this.state.userInfo}
              />
            </Switch>
          </Layout>
        </Layout>
      </BrowserRouter>
    );
  }

  public componentDidMount() {
    document.addEventListener("keydown", this.onKeyDown);
  }

  public componentWillUnmount() {
    document.removeEventListener("keydown", this.onKeyDown);
  }

  public componentWillMount() {
    this.getUserInfo();
  }

  public updateThemeState() {
    // console.log(window.localStorage.getItem("userThemeValue"));
    this.setState({
      userThemeValue: window.localStorage.getItem("userThemeValue"),
    });
  }

  private async getUserInfo() {
    const userInfo = await getUserInfo();
    this.setState({ userInfo });
  }

  private onKeyDown = (ev: KeyboardEvent) => {
    this.kb.push(ev.keyCode);
    if (this.kb.length > kc.length) {
      this.kb.splice(0, this.kb.length - kc.length);
    }

    if (JSON.stringify(this.kb) === JSON.stringify(kc)) {
      const wrapper = document.querySelector("#root");
      if (!wrapper) {
        return;
      }

      if (this.appClass === kcClass) {
        wrapper.classList.remove(this.appClass);
        this.appClass = "";
      } else {
        this.appClass = kcClass;
        wrapper.classList.add(this.appClass);
      }

      this.kb = [];
    }
  };
}

export interface UserInfo {
  login: string;
  permissions: Permissions;
}

export interface Permissions {
  is_admin: boolean;
  is_publisher: boolean;
}

export async function getUserInfo(): Promise<UserInfo | null> {
  const path = `/api/auth/info`;

  const response = await API.get<UserInfo>(path, {});
  if (response.body === undefined) {
    return null;
  }

  return response.body;
}

export default App;
