import { API } from '../lib/api';

export interface Question {
  question_id: string;
  channel_id: string;
  hash_created_by: string;
  // created_on: Date;
  is_answered: boolean;
  // answered_on: Date;
  upvotes: number;
  description: string;
  have_i_voted: boolean;
}

export interface GetQuestionsResponse {
  questions: Question[];
}

export async function createQuestion(channelID: string, description: string): Promise<Question> {
  const path = `/api/question/${channelID}`;

  const response = await API.post<Question>(path, {
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
        description
    }),
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function getQuestions(channelID: string): Promise<Question[]> {
  const path = `/api/question/${channelID}`;

  const response = await API.get<GetQuestionsResponse>(path, {
    headers: {
      'Content-Type': 'application/json',
    }
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body.questions
}

export async function upvoteQuestion(channelID: string, questionID: string): Promise<Question> {
  const path = `/api/question/${channelID}/${questionID}/upvote`;

  const response = await API.post<Question>(path, {
    headers: {
      'Content-Type': 'application/json',
    }
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function deleteAllQuestions(channelID: string): Promise<{}> {
  const path = `/api/question/${channelID}`;

  const response = await API.delete<Question>(path, {
    headers: {
      'Content-Type': 'application/json',
    }
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function deleteQuestion(channelID: string, questionID: string): Promise<{}> {
  const path = `/api/question/${channelID}/${questionID}`;

  const response = await API.delete<Question>(path, {
    headers: {
      'Content-Type': 'application/json',
    }
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function answerQuestion(channelID: string, questionID: string): Promise<Question> {
  const path = `/api/question/${channelID}/${questionID}/answer`;

  const response = await API.post<Question>(path, {
    headers: {
      'Content-Type': 'application/json',
    }
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}