import { API } from '../lib/api';

export interface GetChannelPageInfoResponse {
  playback_url: string,
  auth_token: string;
  channel_title: string,
  channel_description: string,
}

export async function getChannelInfo(channelID: string): Promise<GetChannelPageInfoResponse> {
  const path = `/api/video/channel-info/` + channelID;

  const response = await API.get<GetChannelPageInfoResponse>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return;
  }

  return response.body
}

export interface ChannelInfo {
  channel_id_hash: string;
  channel_id: string;
  channel_title: string;
  channel_description: string;
  bindle_lock_id: string;
  ivs_channel_arn: string;
  ivs_stream_key: string;
  ivs_ingest_url: string;
}

export interface GetChannelsResponse {
  channels: ChannelInfo[];
}

export async function getChannelList(): Promise<ChannelInfo[]> {
  const path = `/api/admin/channel`;

  const response = await API.get<GetChannelsResponse>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body.channels
}

export async function getChannel(channelIDHash: string): Promise<ChannelInfo> {
  const path = `/api/admin/channel/` + channelIDHash;

  const response = await API.get<ChannelInfo>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export interface updateChannelPayload {
  bindle_lock_id: string;
  channel_title: string;
  channel_description: string;
}

export async function updateChannel(channelIDHash: string, payload: updateChannelPayload): Promise<ChannelInfo> {
  const path = `/api/admin/channel/` + channelIDHash;

  const response = await API.put<ChannelInfo>(path, {
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify(payload),
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function createChannel(channelID: string, channelTitle: string, channelDescription: string): Promise<ChannelInfo> {
  const path = `/api/admin/channel`;

  const response = await API.post<ChannelInfo>(path, {
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      channel_id: channelID,
      channel_title: channelTitle,
      channel_description: channelDescription,
    }),
  });

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return response.body
}

export async function deleteChannel(channelIDHash: string): Promise<null> {
  const path = `/api/admin/channel/` + channelIDHash;

  const response = await API.delete<ChannelInfo>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return null;
  }

  return null
}
