import React from "react";
import { Link } from "react-router-dom";
import {
  AlignItems,
  Background,
  Display,
  JustifyContent,
  Layout,
  Title,
  TitleSize,
  FlexWrap,
  ButtonIcon,
  SVGAsset,
  ButtonSize,
  Button,
  ButtonType,
  ButtonIconSize,
  DropDownMenuHeading,
  DropDownMenuInputItem,
  DropDownMenuInputItemType,
  DropDownMenuItem,
  DropDownMenuSeparator,
  DropDownMenuWrapper,
  Position,
  ZIndex,
  Interactable,
  InteractableType,
  BorderRadius,
} from "twitch-core-ui";
import { UserInfo } from "../App";
import { ClickOutDetector } from "./click-out-detector";

export interface InfoBarProps {
  userInfo: UserInfo | null;
  updateThemeState: Function;
  currentTheme: String;
}

export interface InfoBarState {
  menuOpen: Boolean;
  lightDarkVisible: Boolean;
}

export class InfoBar extends React.Component<InfoBarProps, InfoBarState> {
  constructor(props) {
    super(props);

    this.state = { menuOpen: false, lightDarkVisible: false };
    this.handleMenuToggleClick = this.handleMenuToggleClick.bind(this);
    this.handleClickOut = this.handleClickOut.bind(this);
    this.handleSystemThemeUpdate = this.handleSystemThemeUpdate.bind(this);
    this.changeTheme = this.changeTheme.bind(this);
  }

  componentDidMount() {
    // If the user's theme is not set to "system", show the light and dark mode controls
    const userTheme = window.localStorage.getItem("userThemeValue");
    const lightDarkVisible = userTheme === "system" ? false : true;
    this.setState({
      lightDarkVisible: lightDarkVisible,
    });
  }

  handleMenuToggleClick() {
    this.setState({
      menuOpen: !this.state.menuOpen,
    });
  }

  handleClickOut() {
    this.setState({
      menuOpen: false,
    });
  }

  changeTheme(e) {
    const newValue = e.currentTarget.value;
    switch (newValue) {
      case "light":
        window.localStorage.setItem("userThemeValue", "light");
        window.localStorage.setItem("currentThemeValue", "light");
        break;
      case "dark":
        window.localStorage.setItem("userThemeValue", "dark");
        window.localStorage.setItem("currentThemeValue", "dark");
        break;
      default:
        // Case: "system"
        const checked = e.currentTarget.checked;
        this.handleSystemThemeUpdate(checked);
        break;
    }
    this.props.updateThemeState();
  }

  handleSystemThemeUpdate(isActive: Boolean) {
    // e.target: <input class="tw-toggle__input" id="7e017b4c36100584e56a5031c408f989" type="checkbox" data-a-target="tw-toggle">
    if (isActive) {
      // Set the user-selected theme to "system" in local storage
      window.localStorage.setItem("userThemeValue", "system");
      // Hide the "Light" and "Dark" options
      this.setState({
        lightDarkVisible: false,
      });
    } else {
      // Show the "Light" and "Dark" options
      this.setState({
        lightDarkVisible: true,
      });
      if (
        window.matchMedia &&
        window.matchMedia("(prefers-color-scheme: dark)").matches
      ) {
        window.localStorage.setItem("userThemeValue", "dark");
      } else {
        window.localStorage.setItem("userThemeValue", "light");
      }
    }

    // Update the parent theme
    this.props.updateThemeState();
  }

  getCurrentActualTheme() {
    const currentActualTheme = window.localStorage.getItem("currentThemeValue");
    return currentActualTheme;
  }

  public render() {
    let adminSection = <></>;

    if (
      this.props.userInfo &&
      this.props.userInfo.permissions &&
      (this.props.userInfo.permissions.is_admin ||
        this.props.userInfo.permissions.is_publisher)
    ) {
      adminSection = (
        <>
          <DropDownMenuSeparator />

          <DropDownMenuHeading>Admin</DropDownMenuHeading>
          <DropDownMenuItem
            linkTo="/admin"
            figure={{ icon: SVGAsset.NavChannels }}
            onClick={this.handleClickOut}
          >
            All Channels
          </DropDownMenuItem>
        </>
      );
    }

    const lightDarkDisplayValue = this.state.lightDarkVisible
      ? Display.Block
      : Display.Hide;
    const themeSection = (
      <>
        {/* <DropDownMenuSeparator /> */}

        <DropDownMenuHeading>Theme</DropDownMenuHeading>
        <DropDownMenuInputItem
          type={DropDownMenuInputItemType.Toggle}
          label="Match OS Theme"
          value="system"
          checked={this.props.currentTheme === "system"}
          onChange={this.changeTheme}
        />
        <Layout display={lightDarkDisplayValue}>
          <DropDownMenuSeparator />
          <DropDownMenuInputItem
            type={DropDownMenuInputItemType.Radio}
            name="theme-selector"
            label="Light Theme"
            value="light"
            checked={this.props.currentTheme === "light"}
            onChange={this.changeTheme}
          />
          <DropDownMenuInputItem
            type={DropDownMenuInputItemType.Radio}
            name="theme-selector"
            label="Dark Theme"
            value="dark"
            checked={this.props.currentTheme === "dark"}
            onChange={this.changeTheme}
          />
        </Layout>
      </>
    );

    const menuDisplay = this.state.menuOpen ? Display.Flex : Display.Hide;

    const SiteDropMenu = (
      <Layout
        position={Position.Absolute}
        display={menuDisplay}
        attachRight
        attachTop
        margin={{ top: 3 }}
        zIndex={ZIndex.Above}
      >
        <DropDownMenuWrapper>
          {/* <DropDownMenuItem figure={{ icon: SVGAsset.NavChannels }}>
            My Channels
          </DropDownMenuItem> */}

          {themeSection}

          {adminSection}
        </DropDownMenuWrapper>
      </Layout>
    );

    return (
      <Layout zIndex={ZIndex.Above}>
        <Layout
          background={Background.Alt}
          borderTop
          display={Display.Flex}
          alignItems={AlignItems.Center}
          justifyContent={JustifyContent.End}
        >
          <Layout display={Display.Flex} padding={0.5} flexGrow={0}>
            <Interactable
              type={InteractableType.Alpha}
              borderRadius={BorderRadius.Medium}
              linkTo="/"
            >
              <Layout
                display={Display.Flex}
                flexGrow={0}
                flexShrink={0}
                flexWrap={FlexWrap.NoWrap}
                alignItems={AlignItems.Center}
                padding={{
                  left: 0.5,
                  right: 1,
                  top: 0.5,
                  bottom: 0.5,
                }}
              >
                <Layout margin={{ right: 1 }}>
                  <img src="/parsnip.png" width={24} height={24} />
                </Layout>
                <Title size={TitleSize.ExtraSmall}>Parsnip</Title>
              </Layout>
            </Interactable>
          </Layout>

          <Layout flexGrow={1}></Layout>

          <Layout padding={1} display={Display.Flex}>
            <Button
              size={ButtonSize.Small}
              icon={SVGAsset.Plus}
              type={ButtonType.Secondary}
              aria-label="New channel"
              linkTo="/admin/channel/new"
            >
              Create Channel
            </Button>
            <Layout margin={{ left: 1 }} position={Position.Relative}>
              <ClickOutDetector onClickOut={this.handleClickOut}>
                <ButtonIcon
                  size={ButtonIconSize.Small}
                  icon={SVGAsset.More}
                  aria-label="More options"
                  blurAfterClick
                  onClick={this.handleMenuToggleClick}
                />
                {SiteDropMenu}
              </ClickOutDetector>
            </Layout>
          </Layout>
        </Layout>
      </Layout>
    );
  }
}
