import React from "react";
import {
  AlignItems,
  Background,
  Button,
  Display,
  LoadingSpinner,
  Layout,
  TextAlign,
  TextType,
  CoreText,
  SVGAsset,
  InputType,
  Icon,
  JustifyContent,
  Input,
  CoreButtonType,
  CoreButton,
  CoreButtonLabel,
  InteractableType,
  Interactable,
  FlexDirection,
  FlexWrap,
  BorderRadius,
  FontSize,
  Color,
  ButtonType,
  InjectLayout,
  ButtonSize,
} from "twitch-core-ui";
import {
  answerQuestion,
  createQuestion,
  getQuestions,
  upvoteQuestion,
  deleteQuestion,
  Question,
} from "../../../api/api-questions";
import { APIError } from "../../../lib/api";
import { withRouter, RouteComponentProps } from "react-router";
import { UserInfo } from "../../../App";
import "./component.css";

export interface State {
  processing: boolean;
  loadingQuestions: boolean;
  errorMessage: string;
  questions: Array<Question>;
  description: string;
}

export interface PublicProps {
  channelID: string;
  hideInput: boolean;
  userInfo: UserInfo | null;
}

export type RouteProps = RouteComponentProps<{}>;

type Props = PublicProps & RouteProps;

export class QuestionsBodyCore extends React.Component<Props, State> {
  constructor(props) {
    super(props);

    this.state = {
      processing: false,
      loadingQuestions: false,
      errorMessage: "",
      questions: [],
      description: "",
    };
    this.upvoteQuestion.bind(this);
  }

  public componentDidMount() {
    this.getQuestions();
  }

  createQuestion = (e) => {
    // detect pressing enter
    if (e?.keyCode && e?.keyCode != 13) {
      return;
    }

    this.setState({
      processing: true,
    });

    createQuestion(this.props.channelID, this.state.description).then(
      (question) => {
        this.setState((prevState) => ({
          questions: [...prevState.questions, question],
          processing: false,
          description: "",
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  getQuestions = () => {
    this.setState({
      processing: true,
      loadingQuestions: true,
    });

    getQuestions(this.props.channelID).then(
      (questions) => {
        this.setState((prevState) => ({
          questions: questions ? [...prevState.questions, ...questions] : [],
          processing: false,
          loadingQuestions: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
          loadingQuestions: false,
        });
      }
    );
  };

  upvoteQuestion = (questionID) => {
    this.setState({
      processing: true,
    });

    upvoteQuestion(this.props.channelID, questionID).then(
      (question) => {
        this.setState((prevState) => ({
          questions: [
            ...prevState.questions.filter(
              (prevQ) => prevQ.question_id !== question.question_id
            ),
            question,
          ],
          processing: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  answerQuestion = (channelID, questionID) => {
    this.setState({
      processing: true,
    });

    answerQuestion(channelID, questionID).then(
      () => {
        this.setState((prevState) => ({
          questions: [
            ...prevState.questions.filter(
              (prevQ) => prevQ.question_id !== questionID
            ),
          ],
          processing: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  deleteQuestion = (channelID, questionID) => {
    this.setState({
      processing: true,
    });

    deleteQuestion(channelID, questionID).then(
      () => {
        this.setState((prevState) => ({
          questions: [
            ...prevState.questions.filter(
              (prevQ) => prevQ.question_id !== questionID
            ),
          ],
          processing: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  sortQuestions = (a: Question, b: Question): number => {
    if (a["upvotes"] && !b["upvotes"]) {
      return -1;
    }

    if (!a["upvotes"] && b["upvotes"]) {
      return 1;
    }

    if (!a["upvotes"] && !b["upvotes"]) {
      return 0;
    }

    if (a?.upvotes > b?.upvotes) {
      return -1;
    }
    if (a?.upvotes < b?.upvotes) {
      return 1;
    }
    return 0;
  };

  filterAnswered = (question: Question) => {
    return !question.is_answered;
  };

  redirectToPresent = (questionID: string) => {
    this.props.history.push({
      pathname: `/${this.props.channelID}/questions`,
      state: {
        question_id: questionID,
        is_modal_open: false,
      },
    });
  };

  private updateDescription = (event: React.FormEvent<HTMLInputElement>) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.setState({
      description: value,
    });
  };

  public dismissError() {
    this.setState({
      errorMessage: "",
    });
  }

  renderAdminButtons = (question) => {
    return (
      <Layout justifyContent={JustifyContent.End} flexGrow={0} flexShrink={1}>
        <InjectLayout margin={{ right: 1 }}>
          <Button
            type={ButtonType.Secondary}
            size={ButtonSize.Small}
            icon={SVGAsset.Externallink}
            onClick={() => this.redirectToPresent(question.question_id)}
            aria-label="Present to everyone"
          >
            Present to everyone
          </Button>
        </InjectLayout>
        <InjectLayout margin={{ right: 1 }}>
          <Button
            type={ButtonType.Secondary}
            size={ButtonSize.Small}
            icon={SVGAsset.Check}
            onClick={() =>
              this.answerQuestion(this.props.channelID, question.question_id)
            }
            aria-label="Mark question as answered"
          >
            Answered
          </Button>
        </InjectLayout>
        <Button
          type={ButtonType.Secondary}
          size={ButtonSize.Small}
          icon={SVGAsset.Trash}
          onClick={() =>
            this.deleteQuestion(this.props.channelID, question.question_id)
          }
          aria-label="Delete this question"
        >
          Delete
        </Button>
      </Layout>
    );
  };

  renderQuestions = (question: Question) => {
    return (
      <Layout
        background={Background.Alt2}
        elevation={3}
        padding={2}
        display={Display.Flex}
        alignItems={AlignItems.Center}
        margin={{ bottom: 1 }}
      >
        <Layout
          className="question-upvote-interactable"
          margin={{ right: 1 }}
          fullWidth
        >
          <Interactable
            type={InteractableType.Alpha}
            borderRadius={BorderRadius.Medium}
          >
            <Layout
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              flexWrap={FlexWrap.NoWrap}
              flexGrow={0}
              flexShrink={0}
              padding={1}
              alignItems={AlignItems.Center}
              onClick={() => this.upvoteQuestion(question.question_id)}
            >
              <Icon asset={SVGAsset.ArrowUp} />
              <CoreText fontSize={FontSize.Size6} color={Color.Alt} bold>
                {question.upvotes ? question.upvotes : 0}
              </CoreText>
            </Layout>
          </Interactable>
        </Layout>
        <Layout flexGrow={1} flexShrink={1}>
          <CoreText fontSize={FontSize.Size4}>{question.description}</CoreText>
          <Layout margin={{ top: 1 }}>
            {this.renderAdminButtons(question)}
          </Layout>
        </Layout>
      </Layout>
    );
  };

  renderInput = () => (
    <Layout
      background={Background.Alt2}
      elevation={3}
      padding={2}
      display={Display.Flex}
      alignItems={AlignItems.Center}
      margin={{ bottom: 4 }}
    >
      <Layout
        className="question-upvote-interactable"
        fullWidth
        margin={{ right: 1 }}
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Center}
      >
        <Icon asset={SVGAsset.Whisper}></Icon>
      </Layout>
      <Layout flexGrow={1} margin={{ right: 1 }}>
        <Input
          id="submit_question"
          type={InputType.Text}
          onChange={this.updateDescription}
          onKeyDown={this.createQuestion}
          value={this.state.description}
          placeholder="Ask a question..."
          aria-label="Ask a question"
        ></Input>
      </Layout>
      <Layout flexGrow={0} flexShrink={0}>
        <CoreButton
          type={CoreButtonType.Secondary}
          onClick={() => this.createQuestion}
        >
          <CoreButtonLabel aria-label="Submit Question">Submit</CoreButtonLabel>
        </CoreButton>
      </Layout>
    </Layout>
  );

  public render() {
    if (this.state.errorMessage) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>{this.state.errorMessage}</CoreText>
          <Layout margin={{ top: 2 }}>
            <Button onClick={() => this.dismissError()}>OK then</Button>
          </Layout>
        </Layout>
      );
    }

    return (
      <Layout>
        {!this.props.hideInput && this.renderInput()}
        <Layout>
          {this.state.loadingQuestions ? (
            <LoadingSpinner />
          ) : (
            this.state.questions
              .filter(this.filterAnswered)
              .sort(this.sortQuestions)
              .map(this.renderQuestions)
          )}
        </Layout>
      </Layout>
    );
  }
}

export const QuestionsBody = withRouter(QuestionsBodyCore);
