import React from "react";
import {
  Button,
  Display,
  Layout,
  Title,
  TitleSize,
  SVGAsset,
  Icon,
  JustifyContent,
  CoreButton,
  CoreButtonType,
  CoreButtonLabel,
  InjectLayout,
  ButtonType,
} from "twitch-core-ui";
import { withRouter, RouteComponentProps } from "react-router";
import { deleteAllQuestions, Question } from "../../../api/api-questions";
import { APIError } from "../../../lib/api";

export interface PublicProps {
  channelID: string;
}

export interface State {
  processing: boolean;
  loadingQuestions: boolean;
  errorMessage: string;
  questions: Array<Question>;
  description: string;
}

export type RouteProps = RouteComponentProps<{}>;

type Props = PublicProps & RouteProps;

export class QuestionsHeaderCore extends React.Component<Props, State> {
  constructor(props) {
    super(props);

    this.state = {
      processing: false,
      loadingQuestions: false,
      errorMessage: "",
      questions: [],
      description: "",
    };
  }

  deleteAllQuestions = (channelID) => {
    this.setState({
      processing: true,
      loadingQuestions: true,
    });

    deleteAllQuestions(channelID).then(
      () => {
        this.setState(() => ({
          questions: [],
          processing: false,
          loadingQuestions: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
          loadingQuestions: false,
        });
      }
    );
  };

  redirectToPresent = (questionID: string) => {
    this.props.history.push({
      pathname: `/${this.props.channelID}/questions`,
      state: {
        question_id: questionID,
        is_modal_open: false,
      },
    });
  };

  public render() {
    return (
      <Layout
        margin={{ bottom: 2 }}
        display={Display.Flex}
        justifyContent={JustifyContent.Between}
      >
        <Title size={TitleSize.Default}>Questions</Title>
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.End}
          flexGrow={0}
          flexShrink={1}
        >
          <Layout margin={{ right: 1 }}>
            <Button
              type={ButtonType.Secondary}
              onClick={() => this.redirectToPresent(this.props.channelID)}
              icon={SVGAsset.Externallink}
              aria-label="Present questions to viewers"
            >
              Presenter view
            </Button>
          </Layout>
          <Button
            type={ButtonType.Alert}
            icon={SVGAsset.Trash}
            onClick={() => this.deleteAllQuestions(this.props.channelID)}
            aria-label="Delete all questions"
          >
            Delete All
          </Button>
        </Layout>
      </Layout>
    );
  }
}

export const QuestionsHeader = withRouter(QuestionsHeaderCore);
