import * as React from "react";
import videojs from "video.js";
import "video.js/dist/video-js.css";
import "@videojs/themes/dist/fantasy/index.css";
import "./player.css";
import {
  Aspect,
  Background,
  CoreImage,
  CoreText,
  Display,
  FontSize,
  Layout,
  Position,
  ZIndex,
} from "twitch-core-ui";

export interface Props {
  playbackURL: string;
}

export interface State {
  isOffline: Boolean;
}

export class VideoPlayer extends React.Component<Props, State> {
  player: videojs;
  videoNode: React.RefObject<HTMLVideoElement>;

  constructor(props) {
    super(props);
    this.videoNode = React.createRef();

    this.state = {
      isOffline: true,
    };
  }

  componentDidMount() {
    //@ts-ignore
    window.registerIVSTech(videojs);
  }

  public componentDidUpdate(prevProps: Props) {
    if (
      !this.props ||
      !this.props.playbackURL ||
      this.props.playbackURL === prevProps.playbackURL
    ) {
      return;
    }

    // instantiate Video.js
    this.player = videojs(
      this.videoNode.current,
      {
        autoplay: true,
      },
      () => {
        this.player.src(this.props.playbackURL);
        this.player.play();
      }
    );

    // const ivsPlayer = this.player?.getIVSPlayer();
    this.player.on("error", () => {
      const error = this.player.error();
      if (error.code === 4) {
        this.setState({
          isOffline: true,
        });
      }
    });

    this.player.on("loadeddata", () => {
      console.log("loaded data");
      this.setState({
        isOffline: false,
      });
    });
  }

  // destroy player on unmount
  componentWillUnmount() {
    if (this.player) {
      this.player.dispose();
    }
  }

  // wrap the player in a div with a `data-vjs-player` attribute
  // so videojs won't create additional wrapper in the DOM
  // see https://github.com/videojs/video.js/pull/3856
  render() {
    if (this.props.playbackURL) {
      return (
        <Aspect ratio={16 / 9}>
          <Layout
            display={this.state.isOffline ? Display.Hide : Display.Block}
            position={Position.Absolute}
            attachTop
            attachLeft
            fullWidth
            fullHeight
          >
            <div data-vjs-player>
              <video
                ref={this.videoNode}
                className="video-js vjs-theme-fantasy vjs-16-9"
                disablePictureInPicture
                controls
              ></video>
            </div>
          </Layout>
          <Layout
            className="no-select"
            display={this.state.isOffline ? Display.Block : Display.Hide}
            position={Position.Absolute}
            attachTop
            attachLeft
            fullWidth
            fullHeight
          >
            <Layout
              position={Position.Absolute}
              attachRight
              attachTop
              margin={2}
              padding={{ x: 4, y: 2 }}
              background={Background.Overlay}
              zIndex={ZIndex.Above}
            >
              <CoreText fontSize={FontSize.Size4}>OFFLINE</CoreText>
            </Layout>
            <Aspect ratio={16 / 9}>
              <CoreImage src="/offline.png" alt="Offline" />
            </Aspect>
          </Layout>
        </Aspect>
      );
    }
    return <></>;
  }
}
