import * as React from "react";
import {
  CoreText,
  CoreLink,
  Layout,
  Background,
  TextAlign,
  TextType,
  LoadingSpinner,
  Display,
  JustifyContent,
  ButtonType,
  FormGroup,
  Input,
  TextArea,
  ComboInput,
  InputType,
  FormGroupOrientation,
  Button,
  SVGAsset,
  Icon,
  Color,
  AlignItems,
  Column,
  Grid,
  InjectLayout,
  BorderRadius,
} from "twitch-core-ui";
import {
  getChannel,
  updateChannel,
  deleteChannel,
  ChannelInfo,
} from "../../api/api";
import { APIError } from "../../lib/api";
import { Redirect, match } from "react-router-dom";
import { UserInfo } from "../../App";

interface URLParams {
  channel_id_hash: string;
}

interface Props {
  match: match<URLParams>;
  userInfo: UserInfo | null;
}

interface State {
  channel?: ChannelInfo;
  navigate: string;
  errorMessage: string;
  showStreamKey: boolean;
  processing: boolean;
}

export class AdminChannelDetailPage extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    processing: true,
    showStreamKey: false,
    errorMessage: "",
  };

  public render() {
    if (this.state.navigate !== undefined) {
      // Since we're often navigating to the same component
      this.setState({
        navigate: undefined,
      });
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.processing) {
      return <LoadingSpinner />;
    }

    if (this.state.errorMessage) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>{this.state.errorMessage}</CoreText>
          <Layout margin={{ top: 2 }}>
            <Button onClick={() => this.dismissError()}>OK then</Button>
          </Layout>
        </Layout>
      );
    }

    if (!this.state || !this.state.channel) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>Channel is empty... hmm :/</CoreText>
        </Layout>
      );
    }

    let streamKeyInputType = InputType.Password;
    let showButtonText = "Show";

    if (this.state.showStreamKey) {
      streamKeyInputType = InputType.Text;
      showButtonText = "Hide";
    }

    let deleteButton = <></>;

    if (
      this.props.userInfo &&
      this.props.userInfo.permissions &&
      this.props.userInfo.permissions.is_admin
    ) {
      deleteButton = (
        <Layout margin={{ right: 1 }}>
          <Button onClick={() => this.deleteChannel()} type={ButtonType.Alert}>
            Delete
          </Button>
        </Layout>
      );
    }

    const myParnipURL = `${window.location.origin}/${this.state.channel.channel_id}`;

    return (
      <Layout
        className="text-optim-layout"
        fullWidth
        fullHeight
        margin={{ x: "auto" }}
        padding={4}
      >
        <Layout textAlign={TextAlign.Left}>
          <Layout flexGrow={1} margin={{ bottom: 4 }}>
            <InjectLayout margin={{ bottom: 1 }}>
              <CoreText type={TextType.H3}>Channel Details</CoreText>
            </InjectLayout>
            <CoreLink linkTo={myParnipURL} targetBlank>
              {myParnipURL}
            </CoreLink>
          </Layout>

          <Layout margin={{ bottom: 4 }}>
            <Grid>
              <Column cols={6}>
                <Layout margin={{ bottom: 1 }}>
                  <CoreText bold>Channel ID</CoreText>
                  <CoreText color={Color.Alt2}>
                    {this.state.channel.channel_id}
                  </CoreText>
                </Layout>
              </Column>
              <Column cols={6}>
                <Layout margin={{ bottom: 1 }}>
                  <CoreText bold>IVS Channel ARN</CoreText>
                  <CoreText color={Color.Alt2}>
                    {this.state.channel.ivs_channel_arn}
                  </CoreText>
                </Layout>
              </Column>
              <Column cols={6}>
                <Layout margin={{ bottom: 1 }}>
                  <CoreText bold>Ingest Server</CoreText>
                  <CoreText color={Color.Alt2}>
                    {this.formatIngestEndpoint(
                      this.state.channel.ivs_ingest_url
                    )}
                  </CoreText>
                </Layout>
              </Column>
              <Column cols={6}>
                <Layout margin={{ bottom: 1 }}>
                  <FormGroup
                    label="Stream Key"
                    id="ivs_stream_key"
                    orientation={FormGroupOrientation.Vertical}
                  >
                    <ComboInput
                      type={streamKeyInputType}
                      buttonProps={{
                        "aria-label": showButtonText,
                        icon: SVGAsset.Copy,
                        onClick: () => this.copyStreamKey(),
                      }}
                      placeholder="Stream Key"
                      value={this.state.channel.ivs_stream_key}
                      readOnly
                      id="ivs_stream_key"
                    />
                  </FormGroup>
                </Layout>
              </Column>
            </Grid>
          </Layout>

          <Layout
            display={Display.Flex}
            alignItems={AlignItems.Start}
            background={Background.Alt2}
            padding={2}
            elevation={1}
            margin={{ bottom: 4 }}
            borderRadius={BorderRadius.Medium}
          >
            <Layout flexGrow={0} flexShrink={0} margin={{ right: 1 }}>
              <Icon asset={SVGAsset.NotificationInfo} />
            </Layout>
            <Layout fullWidth>
              <CoreText>
                To go live, you will need to use a streaming app (like{" "}
                <CoreLink linkTo="https://obsproject.com/">OBS</CoreLink>). Copy
                and paste the{" "}
                <CoreText type={TextType.Span} bold>
                  ingest server
                </CoreText>{" "}
                and{" "}
                <CoreText type={TextType.Span} bold>
                  stream key
                </CoreText>{" "}
                values into their corresponding places in your streaming app.
              </CoreText>
              {/* TODO: Add short video showing where to paste in OBS */}
              <CoreText>
                <CoreLink>Video tutorial (0:30)</CoreLink>
              </CoreText>
            </Layout>
          </Layout>

          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Title"
              id="channel_title"
              orientation={FormGroupOrientation.Vertical}
              hint="Your channel title will appear at the top of your channel page."
            >
              <Input
                type={InputType.Text}
                placeholder="My cool channel"
                onChange={this.updateChannelTitle}
                value={this.state.channel.channel_title}
                id="channel_title"
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Description"
              id="channel_description"
              hint="Your channel description will appear at the top of your channel page, below the title."
              orientation={FormGroupOrientation.Vertical}
            >
              <TextArea
                placeholder="Welcome to my cool channel! 😎"
                onChange={this.updateChannelDescription}
                value={this.state.channel.channel_description}
                id="channel_description"
              />
            </FormGroup>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Bindle Lock ID"
              id="bindle_lock_id"
              hint="Paste an Amazon Bindle ID here to restrict view permission on this channel."
              orientation={FormGroupOrientation.Vertical}
            >
              <Input
                type={InputType.Text}
                placeholder="Bindle Lock ID"
                onChange={this.updateBindleLockID}
                value={this.state.channel.bindle_lock_id}
                id="bindle_lock_id"
              />
            </FormGroup>
          </Layout>
          <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
            {deleteButton}
            <Layout>
              <Button onClick={() => this.updateChannel()}>Save</Button>
            </Layout>
          </Layout>
        </Layout>
      </Layout>
    );
  }

  private formatIngestEndpoint = (ingest_endpoint: string) => {
    if (ingest_endpoint == "") {
      return "Unknown";
    }

    return "rtmps://" + ingest_endpoint + "/app/";
  };

  private updateBindleLockID = (event: React.FormEvent<HTMLInputElement>) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.state.channel.bindle_lock_id = value;

    this.setState({
      channel: this.state.channel,
    });
  };

  private updateChannelTitle = (event: React.FormEvent<HTMLInputElement>) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.state.channel.channel_title = value;

    this.setState({
      channel: this.state.channel,
    });
  };

  private updateChannelDescription = (
    event: React.FormEvent<HTMLTextAreaElement>
  ) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.state.channel.channel_description = value;

    this.setState({
      channel: this.state.channel,
    });
  };

  public componentDidMount() {
    this.setState({
      processing: true,
    });

    this.getChannel();
  }

  public dismissError() {
    this.setState({
      errorMessage: "",
    });
  }

  public deleteChannel() {
    this.setState({
      processing: true,
    });

    deleteChannel(this.props.match.params.channel_id_hash).then(
      () => {
        this.setState({
          navigate: "/admin",
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  }

  public toggleStreamKey() {
    this.setState({
      showStreamKey: !this.state.showStreamKey,
    });
  }

  public copyStreamKey() {
    navigator.clipboard.writeText(this.state.channel.ivs_stream_key);
  }

  public updateChannel() {
    this.setState({
      processing: true,
    });
    updateChannel(this.props.match.params.channel_id_hash, {
      bindle_lock_id: this.state.channel.bindle_lock_id,
      channel_title: this.state.channel.channel_title,
      channel_description: this.state.channel.channel_description,
    }).then(
      (channel) => {
        this.setState({
          channel,
          showStreamKey: false,
          processing: false,
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  }

  private async getChannel() {
    await getChannel(this.props.match.params.channel_id_hash).then(
      (channel) => {
        this.setState({
          channel,
          showStreamKey: false,
          processing: false,
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  }
}
