import * as React from "react";
import {
  CoreText,
  FlexDirection,
  Layout,
  TextAlign,
  TextType,
  LoadingSpinner,
  Display,
  JustifyContent,
  Table,
  FormGroup,
  Input,
  TextArea,
  ComboInput,
  InputType,
  CoreInteractive,
  FormGroupOrientation,
  TableBody,
  AlignSelf,
  TableCell,
  Button,
  SVGAsset,
  TableHeader,
  TableHeading,
  TableRow,
  Color,
  FontSize,
} from "twitch-core-ui";
import { createChannel } from "../../api/api";
import { APIError } from "../../lib/api";
import { Redirect } from "react-router-dom";

interface State {
  channel_id: string;
  channel_title: string;
  channel_description: string;
  navigate: string;
  errorMessage: string;
  showStreamKey: boolean;
  processing: boolean;
}

export class AdminNewChannelPage extends React.Component<{}, State> {
  public state: State = {
    channel_id: "",
    channel_title: "",
    channel_description: "",
    navigate: undefined,
    processing: false,
    showStreamKey: false,
    errorMessage: "",
  };

  public render() {
    if (this.state.navigate !== undefined) {
      // Since we're often navigating to the same component
      this.setState({
        navigate: undefined,
      });
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.processing) {
      return <LoadingSpinner />;
    }

    if (this.state.errorMessage) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>{this.state.errorMessage}</CoreText>
          <Layout margin={{ top: 2 }}>
            <Button onClick={() => this.dismissError()}>OK then</Button>
          </Layout>
        </Layout>
      );
    }

    return (
      <Layout
        className="text-optim-layout"
        fullWidth
        fullHeight
        padding={4}
        margin={{ x: "auto" }}
      >
        <Layout
          textAlign={TextAlign.Left}
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
        >
          <Layout flexGrow={1} margin={{ bottom: 4 }}>
            <CoreText type={TextType.H3}>Create Channel</CoreText>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Channel ID"
              id="channel_id"
              orientation={FormGroupOrientation.Vertical}
            >
              <Input
                type={InputType.Text}
                placeholder="Channel ID"
                onChange={this.updateChannelID}
                value={this.state.channel_id}
                id="channel_id"
              />
            </FormGroup>
            <Layout margin={{ top: 1 }}>
              <CoreText color={Color.Alt2} sizes={FontSize.Size7}>
                The channel ID will appear in the link to your channel:
                {` ${window.location.origin}/`}
                <CoreText type={TextType.Span} bold>
                  example-channel-id
                </CoreText>
              </CoreText>
            </Layout>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Title"
              id="channel_title"
              orientation={FormGroupOrientation.Vertical}
            >
              <Input
                type={InputType.Text}
                placeholder="Title"
                onChange={this.updateChannelTitle}
                value={this.state.channel_title}
                id="channel_title"
              />
            </FormGroup>
            <Layout margin={{ top: 1 }}>
              <CoreText color={Color.Alt2} sizes={FontSize.Size7}>
                Your channel title will appear at the top of your channel page.
              </CoreText>
            </Layout>
          </Layout>
          <Layout margin={{ bottom: 2 }}>
            <FormGroup
              label="Description"
              id="channel_description"
              orientation={FormGroupOrientation.Vertical}
            >
              <TextArea
                placeholder="Description"
                onChange={this.updateChannelDescription}
                value={this.state.channel_description}
                id="channel_description"
              />
            </FormGroup>
            <Layout margin={{ top: 1 }}>
              <CoreText color={Color.Alt2} sizes={FontSize.Size7}>
                Your channel description will appear at the top of your channel
                page, below the title.
              </CoreText>
            </Layout>
          </Layout>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Start}>
            <Button
              disabled={!this.isValid()}
              onClick={() => this.createChannel()}
            >
              Create Channel
            </Button>
          </Layout>
        </Layout>
      </Layout>
    );
  }

  public dismissError() {
    this.setState({
      errorMessage: "",
    });
  }

  private isValid = (): boolean => {
    if (!this.state.channel_id) {
      return false;
    }

    if (!this.state.channel_title) {
      return false;
    }

    return true;
  };

  private updateChannelID = (event: React.FormEvent<HTMLInputElement>) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.setState({
      channel_id: value,
    });
  };

  private updateChannelTitle = (event: React.FormEvent<HTMLInputElement>) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.setState({
      channel_title: value,
    });
  };

  private updateChannelDescription = (
    event: React.FormEvent<HTMLTextAreaElement>
  ) => {
    let value: string | undefined = event.currentTarget.value;
    if (value === "") {
      value = undefined;
    }

    this.setState({
      channel_description: value,
    });
  };

  public createChannel() {
    this.setState({
      processing: true,
    });

    createChannel(
      this.state.channel_id,
      this.state.channel_title,
      this.state.channel_description
    ).then(
      (channel) => {
        this.setState({
          navigate: "/admin/channel/" + channel.channel_id_hash,
          processing: false,
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  }
}
