import * as React from "react";
import { Link, match, Redirect } from "react-router-dom";
import {
  CoreText,
  Layout,
  TextAlign,
  TextType,
  AlignContent,
  Display,
  Title,
  TitleSize,
  Color,
  InjectLayout,
} from "twitch-core-ui";
import { VideoPlayer } from "../../components/video-player";
import { Questions } from "../../components/questions";
import { getChannelInfo } from "../../api/api";
import { APIError } from "../../lib/api";
import { UserInfo } from "../../App";
import "./channel.css";

interface URLParams {
  channel_id: string;
}

interface Props {
  match: match<URLParams>;
  userInfo: UserInfo | null;
}

interface State {
  loading: boolean;
  errorMessage: string;
  playbackURL: string;
  channelTitle: string;
  channelDescription: string;
}

export class ChannelPage extends React.Component<Props, State> {
  public state: State = {
    loading: true,
    errorMessage: "",
    playbackURL: "",
    channelTitle: "",
    channelDescription: "",
  };

  public render() {
    if (this.state.errorMessage) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>{this.state.errorMessage}</CoreText>
        </Layout>
      );
    }

    return (
      <Layout className="video-container">
        <Layout margin={{ x: "auto", bottom: 4 }} className="text-optim-layout">
          <InjectLayout margin={{ bottom: 1 }}>
            <Title size={TitleSize.Default}>{this.state.channelTitle}</Title>
          </InjectLayout>
          <CoreText color={Color.Alt2}>
            {this.state.channelDescription}
          </CoreText>
        </Layout>
        <VideoPlayer playbackURL={this.state.playbackURL} />
        <Layout
          margin={{ x: "auto", bottom: 4, top: 4 }}
          className="text-optim-layout"
        >
          <Questions
            channelID={this.props.match.params.channel_id}
            userInfo={this.props.userInfo}
          />
        </Layout>
      </Layout>
    );
  }

  public componentDidMount() {
    this.setState({
      loading: true,
    });

    this.getChannelInfo();
  }

  public componentDidUpdate(prevProps: Props) {
    if (
      this.props.match.params.channel_id === prevProps.match.params.channel_id
    ) {
      return;
    }

    this.setState({
      loading: true,
    });

    this.getChannelInfo();
  }

  private async getChannelInfo() {
    await getChannelInfo(this.props.match.params.channel_id).then(
      (channelInfo) => {
        this.setState({
          playbackURL:
            channelInfo.playback_url + "?token=" + channelInfo.auth_token,
          channelTitle: channelInfo.channel_title,
          channelDescription: channelInfo.channel_description,
          errorMessage: "",
          loading: false,
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
        });
      }
    );
  }
}
