import * as React from "react";
import { withRouter, RouteComponentProps, match } from "react-router";
import {
  Button,
  CoreText,
  Layout,
  TextAlign,
  TextType,
  Display,
  Title,
  TitleSize,
  Icon,
  InjectLayout,
  CoreLink,
  ModalLayer,
  SVGAsset,
  JustifyContent,
  ButtonType,
  OverlayRegion,
  Background,
  Color,
  FlexDirection,
  FlexWrap,
  AlignItems,
  FontSize,
  Position,
  ButtonIcon,
  ButtonIconType,
  TransitionType,
} from "twitch-core-ui";
import { getChannelInfo } from "../../api/api";
import { getQuestions, Question } from "../../api/api-questions";
import { APIError } from "../../lib/api";
import { QuestionsBody } from "../../components/questions/questions-body";
import { UserInfo } from "../../App";
import "./component.css";

interface URLParams {
  channel_id: string;
}

interface LocationState {
  question_id: string;
  is_modal_open: boolean | null;
}

interface LocationStateShape {
  state: LocationState;
}

interface PublicProps {
  userInfo: UserInfo | null;
}

interface State {
  channelTitle: string;
  questions: Question[];
  processing: boolean;
  errorMessage: string;
}

export type RouteProps = RouteComponentProps<URLParams>;

type Props = RouteProps & PublicProps;
export class QuestionsPageCore extends React.Component<Props, State> {
  public state: State = {
    channelTitle: "",
    processing: false,
    errorMessage: "",
    questions: [],
  };

  findIndexByQuestionID = (questionID) => {
    const questions = this.getSortedFilteredQuestions();
    const idx = questions.findIndex((q) => q.question_id === questionID);
    return idx === -1 ? 0 : idx;
  };

  getQuestions = (channelID) => {
    this.setState({
      processing: true,
    });

    getQuestions(channelID).then(
      (questions) => {
        this.setState((prevState) => {
          const combinedQuestions = questions
            ? [...prevState.questions, ...questions]
            : [];
          const sortedQuestions = combinedQuestions
            .filter(this.filterAnswered)
            .sort(this.sortQuestions);
          return {
            questions: sortedQuestions,
            processing: false,
          };
        });
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  getChannelInfo = (channelID) => {
    this.setState({
      processing: true,
    });

    getChannelInfo(channelID).then(
      (channelInfo) => {
        this.setState(() => ({
          channelTitle: channelInfo.channel_title,
          processing: false,
        }));
      },
      (reason: APIError) => {
        this.setState({
          errorMessage: reason.message,
          processing: false,
        });
      }
    );
  };

  sortQuestions = (a: Question, b: Question): number => {
    if (a["upvotes"] && !b["upvotes"]) {
      return -1;
    }

    if (!a["upvotes"] && b["upvotes"]) {
      return 1;
    }

    if (!a["upvotes"] && !b["upvotes"]) {
      return 0;
    }

    if (a?.upvotes > b?.upvotes) {
      return -1;
    }
    if (a?.upvotes < b?.upvotes) {
      return 1;
    }
    return 0;
  };

  filterAnswered = (question: Question) => {
    return !question.is_answered;
  };

  decrementIndex = () => {
    const currIdx = this.findIndexByQuestionID(
      this.props.location.state.question_id
    );
    let nextIdx;
    if (currIdx === 0) {
      nextIdx = this.state.questions.length - 1;
    } else {
      nextIdx = currIdx - 1;
    }

    this.props.history.push({
      pathname: `/${this.props.match.params.channel_id}/questions`,
      state: { question_id: this.state.questions[nextIdx].question_id },
    });
  };

  incrementIndex = () => {
    const currIdx = this.findIndexByQuestionID(
      this.props.location.state.question_id
    );
    let nextIdx;
    if (currIdx === 0) {
      nextIdx = this.state.questions.length - 1;
    } else {
      nextIdx = currIdx - 1;
    }

    if (currIdx + 1 === this.state.questions.length) {
      nextIdx = 0;
    } else {
      nextIdx = currIdx + 1;
    }

    this.props.history.push({
      pathname: `/${this.props.match.params.channel_id}/questions`,
      state: { question_id: this.state.questions[nextIdx].question_id },
    });
  };

  setIsModalOpen = (isModalOpen: boolean) => {
    this.props.history.push({
      pathname: `/${this.props.match.params.channel_id}/questions`,
      state: {
        question_id: this.props.location.state?.question_id,
        is_modal_open: isModalOpen,
      },
    });
  };

  getSortedFilteredQuestions = () => {
    return this.state.questions
      .filter(this.filterAnswered)
      .sort(this.sortQuestions);
  };

  public render() {
    if (this.state.errorMessage) {
      return (
        <Layout
          display={Display.Block}
          textAlign={TextAlign.Center}
          padding={4}
        >
          <CoreText type={TextType.H3}>{this.state.errorMessage}</CoreText>
        </Layout>
      );
    }

    const questions = this.getSortedFilteredQuestions();

    let currentQuestion;
    if (this.props.location.state?.question_id) {
      currentQuestion = questions.find((q) => {
        return q.question_id === this.props.location.state.question_id;
      });
    } else {
      currentQuestion = questions[0];
    }
    const upvotes = currentQuestion?.upvotes ? currentQuestion?.upvotes : 0;

    return (
      <Layout
        className="screen-optim-layout"
        padding={4}
        margin={{ x: "auto" }}
        textAlign={TextAlign.Center}
        display={Display.Flex}
        flexWrap={FlexWrap.NoWrap}
        flexDirection={FlexDirection.Column}
        alignItems={AlignItems.Center}
        fullWidth
        fullHeight
      >
        {/* Title Section */}
        <Layout
          margin={{ x: "auto", bottom: 4 }}
          className="text-optim-layout"
          flexGrow={0}
          flexShrink={0}
        >
          <InjectLayout margin={{ bottom: 0.5 }}>
            <Title size={TitleSize.Small}>Current Question</Title>
          </InjectLayout>
          <CoreLink
            linkTo={`/${this.props.match.params.channel_id}`}
            targetBlank
          >
            {this.state.channelTitle}
          </CoreLink>
        </Layout>

        {/* Main question */}
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          flexWrap={FlexWrap.NoWrap}
          flexGrow={1}
          flexShrink={1}
          justifyContent={JustifyContent.Center}
          className="question-present-container"
          fullWidth
          fullHeight
          textAlign={TextAlign.Left}
        >
          <Layout
            background={Background.Accent}
            className="question-present-quote-container"
            margin={{ bottom: 1 }}
          >
            <Layout borderLeft padding={{ x: 2 }}>
              <CoreText fontSize={FontSize.Size3} color={Color.Overlay}>
                {currentQuestion?.description}
              </CoreText>
            </Layout>
          </Layout>
          <Layout
            display={Display.Flex}
            alignItems={AlignItems.Center}
            position={Position.Relative}
            className="question-present-meta-wrapper"
          >
            <CoreText
              color={Color.Link}
              className="question-present-upvote-icon"
              type={TextType.Span}
            >
              <Icon asset={SVGAsset.ArrowUp} fillParent />
            </CoreText>
            <CoreText>
              <CoreText
                type={TextType.Span}
                fontSize={FontSize.Size7}
                bold
              >{`${upvotes} People`}</CoreText>{" "}
              upvoted this
            </CoreText>
          </Layout>
        </Layout>

        {/* Controls */}
        <Layout
          flexGrow={0}
          flexShrink={0}
          className="question-present-controls-wrapper"
        >
          <Layout
            display={
              this.props.location.state.is_modal_open
                ? Display.HideAccessible
                : Display.Block
            }
          >
            <ButtonIcon
              onClick={this.decrementIndex}
              type={ButtonIconType.Secondary}
              icon={SVGAsset.ArrowLeft}
              aria-label="Previous Question"
            />
            <Button
              onClick={() => this.setIsModalOpen(true)}
              type={ButtonType.Secondary}
            >
              Show all questions
            </Button>
            <ButtonIcon
              onClick={this.incrementIndex}
              type={ButtonIconType.Secondary}
              icon={SVGAsset.ArrowRight}
              aria-label="Next Question"
            />
          </Layout>
        </Layout>

        {/* Show Questions Modal */}
        <ModalLayer
          show={this.props.location.state?.is_modal_open}
          transitionType={TransitionType.SlideOverBottom}
          onRequestClose={() => this.setIsModalOpen(false)}
        >
          <Layout
            fullWidth
            fullHeight
            display={Display.Flex}
            alignItems={AlignItems.End}
            justifyContent={JustifyContent.Center}
            padding={4}
          >
            <Layout
              className="text-optim-layout"
              fullWidth
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              flexWrap={FlexWrap.NoWrap}
              justifyContent={JustifyContent.Center}
            >
              <QuestionsBody
                channelID={this.props.match.params.channel_id}
                hideInput
                userInfo={this.props.userInfo}
              />
              <OverlayRegion>
                <Layout
                  margin={{ top: 2 }}
                  display={Display.Flex}
                  justifyContent={JustifyContent.Center}
                >
                  <Button
                    type={ButtonType.Secondary}
                    onClick={() => this.setIsModalOpen(false)}
                  >
                    Close
                  </Button>
                </Layout>
              </OverlayRegion>
            </Layout>
          </Layout>
        </ModalLayer>
      </Layout>
    );
  }

  public componentDidMount() {
    this.getChannelInfo(this.props.match.params.channel_id);
    this.getQuestions(this.props.match.params.channel_id);
    // Enable keyboard listeners
    document.addEventListener("keydown", this.handleKeyDown);
    // this.incrementIndex = this.incrementIndex.bind(this);
    // this.decrementIndex = this.decrementIndex.bind(this);
  }

  public componentDidUpdate(prevProps: Props) {
    if (
      this.props.match.params.channel_id === prevProps.match.params.channel_id
    ) {
      return;
    }

    this.getChannelInfo(this.props.match.params.channel_id);
    this.getQuestions(this.props.match.params.channel_id);
  }

  public componentWillUnmount() {
    document.removeEventListener("keydown", this.handleKeyDown);
  }

  handleKeyDown = (e) => {
    const isInterestingKey =
      e?.key === "ArrowRight" ||
      e?.key === "ArrowLeft" ||
      e?.key === "ArrowUp" ||
      e?.key === "ArrowDown";
    if (e?.key && !isInterestingKey) {
      return;
    } else {
      if (e?.key === "ArrowRight") {
        this.incrementIndex();
      } else if (e?.key === "ArrowLeft") {
        this.decrementIndex();
      } else if (e?.key === "ArrowUp") {
        this.setIsModalOpen(true);
      } else if (e?.key === "ArrowDown") {
        this.setIsModalOpen(false);
      }
    }
  };
}

export const QuestionsPage = withRouter(QuestionsPageCore);
