import * as cdk from '@aws-cdk/core';
import * as s3 from '@aws-cdk/aws-s3';
import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as origins from '@aws-cdk/aws-cloudfront-origins';
import * as lambda from '@aws-cdk/aws-lambda';
import * as iam from '@aws-cdk/aws-iam';
import * as apigateway from '@aws-cdk/aws-apigatewayv2';
import * as apigatewayintegrations from '@aws-cdk/aws-apigatewayv2-integrations';
import * as sam from '@aws-cdk/aws-sam';

export interface StackProps {
  cdkProps: cdk.StackProps;
  bucketName: string
}

export class CdkStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: StackProps) {
    super(scope, id, props.cdkProps);

    // Lets start by creating the S3 bucket where we will store the captured clips.
    let bucket = new s3.Bucket(this, "qlips-lambda-bucket", {
      bucketName: props.bucketName,
    })

    // Instead of exposing an S3 bucket publicly (bad), this creates a CloudFront distribution with an S3 Origin Access Identity.
    let cfDistro = new cloudfront.Distribution(this, "qlips-cdn", {
      defaultBehavior: {
        origin: new origins.S3Origin(bucket, {})
      }
    })

    // Qlips depends on another published lambda layer for FFmpeg. Lets deploy this layer as a nested stack into our application.
    // Once that is done we capture the output layer version and create a new Lambda Layer based on it for our application.
    let ffmpegLayerSamApp = new sam.CfnApplication(this, 'FFmpegLayer', {
      location: {
        applicationId: 'arn:aws:serverlessrepo:us-east-1:145266761615:applications/ffmpeg-lambda-layer',
        semanticVersion: '1.0.0'
      }
    });

    let layerVersionArn = cdk.Stack.of(this).resolve(ffmpegLayerSamApp.getAtt('Outputs.LayerVersion'))
    let ffmpegLayer = lambda.LayerVersion.fromLayerVersionArn(this, 'FFmpegLayerVersion', layerVersionArn)

    // We can now go ahead and create the Lambda that will generate qlips. We pass in the bucket and url so it can return the proper clip path.
    let lambdaFunction = new lambda.Function(this, "qlips-backend", {
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 512,
      timeout: cdk.Duration.minutes(5),
      handler: 'bin/qlips',
      layers: [ffmpegLayer],
      code: lambda.Code.fromAsset('../bin/qlips.zip'),
      environment: {
        "bucket": bucket.bucketName,
        "url": cfDistro.domainName
      }
    })

    // The Lambda function needs the ability to write files to S3, so lets grant it PUT access to our bucket.
    lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
      sid: "qlipswriteaccess",
      effect: iam.Effect.ALLOW,
      actions: [
        "s3:PutObject"
      ],
      resources: [
        bucket.bucketArn,
        bucket.arnForObjects("*"),
      ]
    }))

    // Our frontend needs a way of calling the Lambda function so lets use a API Gateway in front of the Lambda with a proxy integration for the routes.
    let gw = new apigateway.HttpApi(this, "qlips-gateway", {
      apiName: "qlips-gw",
    })

    gw.addRoutes({
      path: '/clip',
      methods: [ apigateway.HttpMethod.GET ],
      integration: new apigatewayintegrations.LambdaProxyIntegration({
        handler: lambdaFunction,
      }),
    });

  }
}
