// Copyright 2013, zhangpeihao All rights reserved.

package rtmp

import (
	"encoding/binary"
	"errors"
)

// Encode header into io.Writer
func (header *Header) Write(wbuf Writer) (n int, err error) {
	// Write fmt & Chunk stream ID
	switch {
	case header.ChunkStreamID <= 63:
		err = wbuf.WriteByte(byte((header.Fmt << 6) | byte(header.ChunkStreamID)))
		if err != nil {
			return
		}
		n++
	case header.ChunkStreamID <= 319:
		err = wbuf.WriteByte(header.Fmt << 6)
		if err != nil {
			return
		}
		n++
		err = wbuf.WriteByte(byte(header.ChunkStreamID - 64))
		if err != nil {
			return
		}
		n++
	case header.ChunkStreamID <= 65599:
		err = wbuf.WriteByte((header.Fmt << 6) | 0x01)
		if err != nil {
			return
		}
		n++
		tmp := uint16(header.ChunkStreamID - 64)
		err = binary.Write(wbuf, binary.BigEndian, &tmp)
		if err != nil {
			return
		}
		n += 2
	default:
		return n, errors.New("Unsupport chunk stream ID large then 65599")
	}
	tmpBuf := make([]byte, 4)
	var m int
	switch header.Fmt {
	case HEADER_FMT_FULL:
		// Timestamp
		binary.BigEndian.PutUint32(tmpBuf, header.Timestamp)
		m, err = wbuf.Write(tmpBuf[1:])
		if err != nil {
			return
		}
		n += m
		// Message Length
		binary.BigEndian.PutUint32(tmpBuf, header.MessageLength)
		m, err = wbuf.Write(tmpBuf[1:])
		if err != nil {
			return
		}
		n += m
		// Message Type
		err = wbuf.WriteByte(header.MessageTypeID)
		if err != nil {
			return
		}
		n++
		// Message Stream ID
		err = binary.Write(wbuf, binary.LittleEndian, &(header.MessageStreamID))
		if err != nil {
			return
		}
		n += 4
	case HEADER_FMT_SAME_STREAM:
		// Timestamp
		binary.BigEndian.PutUint32(tmpBuf, header.Timestamp)
		m, err = wbuf.Write(tmpBuf[1:])
		if err != nil {
			return
		}
		n += m
		// Message Length
		binary.BigEndian.PutUint32(tmpBuf, header.MessageLength)
		m, err = wbuf.Write(tmpBuf[1:])
		if err != nil {
			return
		}
		n += m
		// Message Type
		err = wbuf.WriteByte(header.MessageTypeID)
		if err != nil {
			return
		}
		n++
	case HEADER_FMT_SAME_LENGTH_AND_STREAM:
		// Timestamp
		binary.BigEndian.PutUint32(tmpBuf, header.Timestamp)
		m, err = wbuf.Write(tmpBuf[1:])
		if err != nil {
			return
		}
		n += m
	case HEADER_FMT_CONTINUATION:
		// do nothing
		return
	}

	// Type 3 chunks MUST NOT have Extended timestamp????
	// Todo: Test with FMS
	// if header.Timestamp >= 0xffffff && header.Fmt != HEADER_FMT_CONTINUATION {
	if header.Timestamp >= 0xffffff {
		// Extended Timestamp
		err = binary.Write(wbuf, binary.BigEndian, &(header.ExtendedTimestamp))
		if err != nil {
			return
		}
		n += 4
	}
	return
}

func (header *Header) RealTimestamp() uint32 {
	if header.Timestamp >= 0xffffff {
		return header.ExtendedTimestamp
	}
	return header.Timestamp
}

func (header *Header) Dump(name string) {
	// logger.ModulePrintf(logHandler, log.LOG_LEVEL_DEBUG,
	// 	"Header(%s){Fmt: %d, ChunkStreamID: %d, Timestamp: %d, MessageLength: %d, MessageTypeID: %d, MessageStreamID: %d, ExtendedTimestamp: %d}\n", name,
	// 	header.Fmt, header.ChunkStreamID, header.Timestamp, header.MessageLength,
	// 	header.MessageTypeID, header.MessageStreamID, header.ExtendedTimestamp)
}
