package main

import (
	"code.justin.tv/event-engineering/rtmp/pkg/rtmpdump"
	"fmt"
	"github.com/spf13/cobra"
	"os"
	"os/signal"
	"syscall"
)

func init() {
	rootCmd.AddCommand(rtmpDumpCommand)
	rtmpDumpCommand.PersistentFlags().StringP("input", "i", "", "Input RTMP URL")
	rtmpDumpCommand.PersistentFlags().StringP("output", "o", "", "Output file")
}

var rtmpDumpCommand = &cobra.Command{
	Use:   "dump",
	Short: "Dump the RTMP to file until told to stop",
	Run: func(cmd *cobra.Command, args []string) {
		input, err := cmd.Flags().GetString("input")
		if err != nil {
			fmt.Println("Unable to parse input ", err)
			os.Exit(1)
		}

		output, err := cmd.Flags().GetString("output")
		if err != nil {
			fmt.Println("Unable to parse input ", err)
			os.Exit(1)
		}

		if input == "" {
			fmt.Println("No input specified, use -i")
			os.Exit(1)
		}

		if output == "" {
			fmt.Println("No output specified, use -o")
			os.Exit(1)
		}

		errCh, ender, finalised := rtmpdump.Dump(input, output, logger)
		fmt.Printf("Dumping to %v\n", output)

		osSig := make(chan os.Signal, 1)
		signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

		go func() {
			err := <-errCh
			fmt.Printf("Error dumping RTMP %v\n", err)
			os.Exit(1)
		}()

		sig := <-osSig
		fmt.Printf("Finalising output after receiving signal %v\n", sig)
		close(ender)

		<-finalised

		fmt.Printf("Saved output to %v\n", output)
	},
}
