package flvlooper

import (
	flv "code.justin.tv/video/goflv"
	"github.com/pkg/errors"
	"os"
)

type File struct {
	file *os.File
	pos  int64
}

func Create(file *os.File) (flvFile *File, err error) {
	if _, err = file.Seek(0, 0); err != nil {
		return nil, errors.New("Could not seek to 0")
	}

	flvFile = &File{
		file: file,
	}

	// Read flv header
	flvHeader := make([]byte, flv.HEADER_LEN)

	if _, err = file.ReadAt(flvHeader, flvFile.pos); err != nil {
		return nil, errors.New("FLV header not present")
	}

	if flvHeader[0] != 'F' ||
		flvHeader[1] != 'L' ||
		flvHeader[2] != 'V' {
		return nil, errors.New("FLV Header malformed")
	}

	flvFile.pos += int64(len(flvHeader))

	return
}

func (flvFile *File) Close() {
	flvFile.file.Close()
}

func (flvFile *File) ReadTag() (header *flv.TagHeader, data []byte, err error) {
	var tagStartPos = flvFile.pos

	tmpBuf := make([]byte, 4)
	header = &flv.TagHeader{}
	// Read tag header
	if _, err = flvFile.file.ReadAt(tmpBuf[3:], flvFile.pos); err != nil {
		return
	}
	header.TagType = tmpBuf[3]
	flvFile.pos += int64(len(tmpBuf[3:]))

	// Read tag size
	if _, err = flvFile.file.ReadAt(tmpBuf[1:], flvFile.pos); err != nil {
		flvFile.pos = tagStartPos
		return
	}
	header.DataSize = uint32(tmpBuf[1])<<16 | uint32(tmpBuf[2])<<8 | uint32(tmpBuf[3])
	flvFile.pos += int64(len(tmpBuf[1:]))

	// Read timestamp
	if _, err = flvFile.file.ReadAt(tmpBuf, flvFile.pos); err != nil {
		flvFile.pos = tagStartPos
		return
	}
	header.Timestamp = uint32(tmpBuf[3])<<32 + uint32(tmpBuf[0])<<16 + uint32(tmpBuf[1])<<8 + uint32(tmpBuf[2])
	flvFile.pos += int64(len(tmpBuf))

	// Read stream ID
	if _, err = flvFile.file.ReadAt(tmpBuf[1:], flvFile.pos); err != nil {
		flvFile.pos = tagStartPos
		return
	}
	flvFile.pos += int64(len(tmpBuf[1:]))

	// Read data
	data = make([]byte, header.DataSize)
	if _, err = flvFile.file.ReadAt(data, flvFile.pos); err != nil {
		flvFile.pos = tagStartPos
		return
	}
	flvFile.pos += int64(len(data))

	// Read previous tag size
	if _, err = flvFile.file.ReadAt(tmpBuf, flvFile.pos); err != nil {
		flvFile.pos = tagStartPos
		return
	}
	flvFile.pos += int64(len(tmpBuf))

	return
}

func (flvFile *File) LoopBack() {
	flvFile.pos = flv.HEADER_LEN
}
