package rtmp

import (
	"fmt"
	"github.com/pkg/errors"
	"net/url"
	"strings"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	rtmp "code.justin.tv/event-engineering/rtmp/pkg/common"
)

// Player will play the RTMP stream for the supplied URL and send data to the MediaStream passed to PlayRTMP()
type Player struct {
	URL       *url.URL
	Conn      gortmp.BasicConn
	ms        gortmp.MediaStream
	errCh     chan error
	tm        *rtmp.TransactionMap
	logger    logging.Logger
	closed    bool
	playing   chan (interface{})
	isPlaying bool
}

// NewPlayer creates a new rtmp Player
func NewPlayer(errCh chan error, logger logging.Logger) *Player {
	return &Player{
		errCh:  errCh,
		logger: logger,
	}
}

// Play will connect to the supplied RTMP URL, sending data from the server to the supplied mediastream
func (p *Player) Play(rtmpURL string, ms gortmp.MediaStream) error {
	parsedURL, conn, appName, err := rtmp.Connect(rtmpURL, 0, 0)
	if err != nil {
		return err
	}

	p.URL = parsedURL
	p.Conn = conn
	p.tm = rtmp.NewTransactionMap()
	p.ms = ms
	p.playing = make(chan interface{})

	// first just send the connect command
	connectCmd := rtmp.DefaultConnectCommand(appName, p.URL.String(), p.tm.New(p.connectResult))

	p.logger.Debugf("Writing connect msg: %+v", connectCmd)
	if err := rtmp.Send(p.Conn, connectCmd); err != nil {
		return err
	}

	if err := rtmp.Send(p.Conn, gortmp.SetChunkSizeMessage{Size: 4096}); err != nil {
		return err
	}

	go p.listen()

	select {
	case <-p.playing:
		return nil
	case <-time.After(5 * time.Second):
		close(p.playing)
		p.Close()
		return errors.New("Playback timed out after 5 seconds")
	}
}

// listen for responses from the server
func (p *Player) listen() {
	for {
		if p.closed {
			return
		}
		raw, err := p.Conn.Read()
		if err != nil {
			if !p.closed {
				p.errCh <- errors.Wrap(err, "Failed to read from connection")
			}
			return
		}

		if err := p.handleMessage(raw); err != nil {
			if !p.closed {
				p.errCh <- errors.Wrap(err, "Failed to handle message")
			}
			return
		}
	}
}

func (p *Player) handleMessage(raw *gortmp.RawMessage) error {
	// Trigger playing on the first video tag
	if raw.Type == gortmp.VIDEO_TYPE && !p.isPlaying {
		p.isPlaying = true
		p.logger.Info("Player is playing")
		p.playing <- "woopwoop"
	}

	msg, err := rtmp.ParseMessage(raw)
	if err != nil {
		return err
	}

	switch msg := msg.(type) {
	case *gortmp.RawMessage, *gortmp.RawCommand:
		// grab the raw message from the command and publish it
		return p.ms.Publish(&gortmp.FlvTag{
			Type:      raw.Type,
			Timestamp: raw.Timestamp,
			Size:      uint32(raw.Data.Len()),
			Bytes:     raw.Data.Bytes(),
		})
	case gortmp.ResultCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case gortmp.ErrorCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case gortmp.OnStatusCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	default:
		//p.logger.Debugf("What's this: %#v", msg)
	}
	return nil
}

// Close closes the player and terminates the connection
func (p *Player) Close() {
	p.closed = true

	if p.Conn != nil {
		err := p.Conn.Close()
		if err != nil {
			p.logger.Warn("Error closing input connection", err)
		}
	}
}

// IsClosed returns true if the player has been closed
func (p *Player) IsClosed() bool {
	return p.closed
}

func (p *Player) connectResult(msg gortmp.Message) error {
	// Handle the result of the connect command, sending a CreateStream command
	cmd := gortmp.CreateStreamCommand{TransactionID: p.tm.New(p.createStreamResult)}
	return rtmp.Send(p.Conn, cmd)
}

func (p *Player) createStreamResult(msg gortmp.Message) error {
	// Handle the result of the CreateStream command
	result, ok := msg.(gortmp.ResultCommand)
	if !ok {
		return fmt.Errorf("Got invalid result command for CreateStream: %#v", msg)
	}

	p.logger.Debugf("CreateStreamResult: %#v", result)

	// Grab the stream ID
	streamID, ok := result.Info.(float64)
	if !ok {
		return fmt.Errorf("Invalid result command for CreateStream: %#v", result)
	}

	// Grab the stream name from the URL
	path := strings.Split(p.URL.Path, "/")
	if len(path) < 3 {
		return fmt.Errorf("Unable to parse stream name: %s", p.URL.Path)
	}

	streamName := strings.Join(path[2:], "/")

	// Send the Play command
	cmd := gortmp.PlayCommand{
		StreamID: uint32(streamID),
		Name:     streamName,
	}

	return rtmp.Send(p.Conn, cmd)
}
