package rtmptail

import (
	"context"
	"github.com/pkg/errors"
	"time"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	player "code.justin.tv/event-engineering/rtmp/pkg/player"
	flv "code.justin.tv/video/goflv"
	rtmplog "code.justin.tv/event-engineering/gortmp/pkg/log"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
)

// Tail opens a connection to the supplied RTMP server, tailing a maximum of limit seconds of data
// the ender chan should be signalled to stop recording and write the buffer to File
// finalised is signalled when the file has been written
func Tail(url string, outputFile string, limit int, logger logging.Logger) (errCh chan error, ender chan (interface{}), finalised chan (interface{})) {
	maxDuration := time.Duration(limit) * time.Second

	rtmplog.SetLogLevel(rtmplog.LogError)

	player := player.NewPlayer(errCh, logger)

	ctx := context.Background()
	ms := gortmp.NewMediaStream(ctx, "dumpthis")
	err := player.Play(url, ms)
	if err != nil {
		errCh <- errors.Wrap(err, "Could not connect to stream")
		return
	}

	ender = make(chan interface{}, 1)
	finalised = make(chan interface{}, 1)
	errCh = make(chan error)

	// Grab rtmp messages from the connection and buffer them
	go tailLoop(ms, player, outputFile, maxDuration, logger, errCh, ender, finalised)

	return errCh, ender, finalised
}

func tailLoop(ms gortmp.MediaStream, player *player.Player, outputFile string, maxDuration time.Duration, logger logging.Logger, errCh chan error, ender chan (interface{}), finalised chan (interface{})) {
	var flvTags []*gortmp.FlvTag
	var startTagIndex int
	var initialTimestamp uint32
	var durationReached bool
	var videoSequenceHeader *gortmp.FlvTag
	var audioSequenceHeader *gortmp.FlvTag

	// Create the output file here so we can bomb out if we don't have permission rather than waiting until
	// finalisation just to find out you can't even write to the file
	flvFile, err := flv.CreateFile(outputFile)

	defer flvFile.Close()

	if err != nil {
		errCh <- errors.Wrap(err, "Could not open output file")
		return
	}

	tags, err := ms.Subscribe()
	if err != nil {
		errCh <- errors.Wrap(err, "Could not subscribe to media stream")
		return
	}

	for {
		select {
		case <-ender:
			player.Close()
			finalise(videoSequenceHeader, audioSequenceHeader, flvTags, startTagIndex, flvFile, errCh, finalised)
			return

		case tag := <-tags:
			// We'll use this to measure the duration of our tail
			if initialTimestamp == 0 {
				initialTimestamp = tag.Timestamp
			}

			// Grab the video sequence header, we'll need it later
			if tag.Type == gortmp.VIDEO_TYPE && videoSequenceHeader == nil {
				header, err := tag.GetVideoHeader()
				if err != nil {
					logger.Debug("Failed to parse Video header, ignoring")
					continue
				}

				if header.AVCPacketType == 0 {
					videoSequenceHeader = tag
					continue
				}
			}

			// Grab the AAC sequence header, we'll need it later
			if tag.Type == gortmp.AUDIO_TYPE && audioSequenceHeader == nil {
				header, err := tag.GetAudioHeader()
				if err != nil {
					logger.Debug("Failed to parse Audio header, ignoring")
					continue
				}
				if header.AACPacketType == 0 {
					audioSequenceHeader = tag
					continue
				}
			}

			// Work out if we should just be grabbing more data, or replacing data we already have
			// we do it like this in case we get any tags with 0 timestamp in the middle of the stream
			if !durationReached {
				duration := time.Duration(tag.Timestamp-initialTimestamp) * time.Millisecond
				durationReached = duration >= maxDuration
			}

			if !durationReached {
				// Just append data
				flvTags = append(flvTags, tag)
			} else {
				// Start looping around our slice replacing the data that's there

				// I'm not sure if the following is possible (multiple sequence headers in a stream), but it's here just in case :D
				// If we're about to replace a sequence header we need to update our stored header so that this new header is used
				// to initialise the output file instead of the old one
				if flvTags[startTagIndex].Type == gortmp.VIDEO_TYPE {
					header, err := flvTags[startTagIndex].GetVideoHeader()
					if err == nil && header.AVCPacketType == 0 {
						logger.Debug("Replacing video sequence header")
						videoSequenceHeader = flvTags[startTagIndex]
					}
				}

				if flvTags[startTagIndex].Type == gortmp.AUDIO_TYPE {
					header, err := flvTags[startTagIndex].GetAudioHeader()
					if err == nil && header.AACPacketType == 0 {
						logger.Debug("Replacing audio sequence header")
						audioSequenceHeader = flvTags[startTagIndex]
					}
				}

				// Replace the existing tag
				flvTags[startTagIndex] = tag

				// Increment or reset the start index
				if startTagIndex == len(flvTags)-1 {
					startTagIndex = 0
				} else {
					startTagIndex++
				}
			}
		}
	}
}

func finalise(videoSequenceHeader *gortmp.FlvTag, audioSequenceHeader *gortmp.FlvTag, flvTags []*gortmp.FlvTag, startTagIndex int, flvFile *flv.File, errCh chan error, finalised chan (interface{})) {
	// Start with the headers
	orderedTags := []*gortmp.FlvTag{videoSequenceHeader, audioSequenceHeader}

	// Append all the rtmp messages from the start index to the end
	orderedTags = append(orderedTags, flvTags[startTagIndex:]...)

	// Append all the rtmp messages from the start of the slice to the startTagIndex
	orderedTags = append(orderedTags, flvTags[0:startTagIndex]...)

	// The header timestamps will be from when we started recording, in order for duration to be calcuated correctly
	// we set them to be the lowest timestamp we have in our output, tags should be in order so just grab the first one we find
	// 2: because we already added the audio/video headers and we don't want to include those in our loop
	for _, tag := range orderedTags[2:] {
		if tag.Timestamp > 0 {
			videoSequenceHeader.Timestamp = tag.Timestamp
			audioSequenceHeader.Timestamp = tag.Timestamp
			break
		}
	}

	// Write the ordered messages to the file
	for _, tag := range orderedTags {
		err := flvFile.WriteTag(tag.Bytes, tag.Type, tag.Timestamp)
		if err != nil {
			// We're going to bomb out in this scenario because we wouldn't be able to trust the output otherwise
			errCh <- errors.Wrap(err, "Error writing flv tag")
			return
		}
	}

	// Signal that we're done
	close(finalised)
}
