const https = require('https');
const qs = require('querystring');
const AWS = require('aws-sdk');
const kms = new AWS.KMS();
const env_vars = ['bridge_verification_token', 'bridge_bot_access_token'];

let variables_decrypted = false;
let envs = {};
let bot_user_id = process.env['bridge_bot_user_id'];
let alert_channel_id = process.env['bridge_alert_channel_id'];
let eng_subteam_id = process.env['bridge_eng_subteam_id'];

let bot_help_regex = new RegExp('^<@' + bot_user_id + '> help.*$');

function handleMessageEvent(event, callback) {
    if (event.hidden) {
        return;
    }

    // Check if this message is actually meant for us
    if (bot_help_regex.test(event.text)) {
        slackPost('chat.postMessage', { channel: alert_channel_id, unfurl_links: true, text: '<!subteam^' + eng_subteam_id +  '> https://twitchbridge.slack.com/archives/' + event.channel + '/p' + event.ts.replace('.', '') })
    }
}

function processEvent(event, context, callback) {
    console.log(JSON.stringify(event));

    // Events
    if (event.context['resource-path'] === '/event') {
        // Handle verification
        if (event.data.type === 'url_verification') {
            callback(null, { body: event.data.challenge });
        } else {
            switch(event.data.event.type){
                case 'message':
                    handleMessageEvent(event.data.event, callback);
                    return;
            }
            callback(null, { body: 'OK' });
        }
    } else if (event.context['resource-path'] === '/notify') {
        if (event.data.token !== envs['bridge_verification_token']) {
            callback('400', { body: 'Invalid Token' })
        }
        // Notify all channels that the bot is in
        //slackPost('users.info', { user: event.data.user_id }, function(user) {
        //    console.log(user);

            slackPost('conversations.list', { exclude_archived: true }, function(response) {
                if (!response.ok) {
                   callback(response.error, { body: response.error });
                   return;
                }

                var messages = []

                for (var i = 0; i < response.channels.length; i++) {
                    var channel = response.channels[i];
                    // Alert public channels and shared channels
                    if (channel.is_channel || channel.is_ext_shared) {
                      messages.push({ channel: channel.id, text: event.data.text })
                    }
                }

                // Return early so we don't hit the slack timeout of 3s
                callback(null, { body: 'Notifying ' + messages.length + ' channels' });

                for (var i = 0; i < messages.length; i++) {
                    slackPost('chat.postMessage', messages[i])
                }
            });
       // });
    }
}

var slackPost = function(path, params, callback) {
    if (!params) {
        params = {};
    }

    params['token'] = envs['bridge_bot_access_token'];

    var post_data = qs.stringify(params);

    var req = https.request({
        hostname: 'slack.com',
        port: 443,
        path: '/api/' + path,
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
            'Content-Length': Buffer.byteLength(post_data)
        }
    }, function (res) {
        res.setEncoding('utf8');
        var data = '';

        res.on('data', function (chunk) {
          data += chunk;
        });

        res.on('end', function () {
            if (callback) {
                console.log(data);
                callback(JSON.parse(data));
            }
        });
    });

    req.write(post_data);
    req.end();
}

var decryptEnvVariable = function(name) {
    return new Promise(function(resolve, reject) {
        kms.decrypt({ CiphertextBlob: new Buffer(process.env[name], 'base64') }, (err, data) => {
            if (err) {
                console.log('Decrypt error:', err);
                reject(err);
            }
            resolve({ name: name, value: data.Plaintext.toString('ascii') });
        });
    });
}

exports.handler = (event, context, callback) => {
    if (variables_decrypted) {
        processEvent(event, context, callback);
    } else {
        var promises = [], i;
        for (i = 0; i < env_vars.length; i++) {
            promises.push(decryptEnvVariable(env_vars[i]))
        }

        Promise.all(promises).then(function(results) {
            for (i = 0; i < results.length; i++) {
                envs[results[i].name] = results[i].value;
            }

            variables_decrypted = true;
            processEvent(event, context, callback);
        })
    }
};
