package main

import (
	"bytes"
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"time"

	"code.justin.tv/jleroux/go-isengard2/service"
	"code.justin.tv/jleroux/go-isengard2/service/isengard"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	v4 "github.com/aws/aws-sdk-go/aws/signer/v4"
)

const apiURL = "https://api.us-west-2.starfruit.io/"
const rateLimit = time.Millisecond * 200

var channels = map[string]string{
	"1080p60":        "",
	"1080p30":        "",
	"720p60":         "",
	"720p30":         "",
	"vertical-video": "",
	"square-video":   "",
}

var rebuild = flag.Bool("rebuild", false, "Remove all stream keys and create them again")

func init() {
	flag.Parse()
}

func main() {
	svc, err := service.NewIsengardService()
	if err != nil {
		log.Fatalf("Failed to initiate isengard client (you probably need to run mwinit): %v", err)
	}
	output, _ := svc.GetAssumeRoleCredentials(&isengard.GetAssumeRoleCredentialsInput{
		AWSAccountID: aws.String("893648527354"),
		IAMRoleName:  aws.String("Admin"),
	})

	result := &service.AssumeRoleResult{}
	_ = json.Unmarshal([]byte(*output.AssumeRoleResult), result)

	creds := credentials.NewStaticCredentials(result.Credentials.AccessKeyID, result.Credentials.SecretAccessKey, result.Credentials.SessionToken)

	// Create v4.Signer to authenticate API requests
	signer := v4.NewSigner(creds)

	sfClient := &client{
		httpClient: &http.Client{},
		signer:     signer,
		lastReq:    time.Now(),
	}

	for channel := range channels {
		// Create the video channel ID, this will fail if the channel already exists, which is fine, so just log the error and continue
		fmt.Printf("Creating Video Channel ID [%v]\n", channel)
		_, err := sfClient.CreateVideoChannelID(channel)
		if err != nil {
			fmt.Println("Failed to create video channel:", err)
		}

		// List the stream keys for the channel, if a key already exists then use it, else create one
		keys, err := sfClient.ListStreamKeys(channel)
		if err != nil {
			fmt.Println("Failed to list stream keys:", err)
		}

		if *rebuild {
			for _, key := range keys {
				err = sfClient.DeleteStreamKey(key)
				if err != nil {
					log.Fatalf("Failed to delete stream key %v - %v\n", key, err)
				}
			}

		} else if len(keys) > 0 {
			channels[channel] = keys[0]
			continue
		}

		// Create a Stream Key
		fmt.Println("Creating Stream Key")
		key, err := sfClient.CreateStreamKey(channel)
		if err != nil {
			log.Fatalln("Failed to perform request: ", err)
		}
		channels[channel] = key
	}

	for channel, streamKey := range channels {
		fmt.Printf("%v %v\n", channel, streamKey)
	}
}

type errResp struct {
	Code    int    `json:"Code"`
	Error   string `json:"Error"`
	Message string `json:"Message"`
}

type client struct {
	httpClient *http.Client
	signer     *v4.Signer
	lastReq    time.Time
}

type listStreamKeysResp struct {
	StreamKeys []streamKey `json:"StreamKeys"`
}

// There's a bunch of other fields here but we only care about Name for our purposes
type streamKey struct {
	Name string `json:"Name"`
}

func (c *client) ListStreamKeys(channelID string) ([]string, error) {
	params := map[string]string{
		"VideoChannelID": channelID,
	}
	resp, status, err := c.performAPIRequest("ListStreamKeys", params)

	if err != nil {
		return nil, err
	}

	if status == http.StatusNotFound {
		return nil, nil
	}

	if status != http.StatusOK {
		return nil, errors.New(resp)
	}

	lskr := &listStreamKeysResp{}
	err = json.Unmarshal([]byte(resp), lskr)
	if err != nil {
		return nil, err
	}

	keys := make([]string, 0)

	for _, sk := range lskr.StreamKeys {
		keys = append(keys, sk.Name)
	}

	return keys, nil
}

type createStreamKeyResp struct {
	StreamKey streamKey `json:"StreamKey"`
}

func (c *client) CreateStreamKey(channelID string) (string, error) {
	params := map[string]string{
		"VideoChannelID": channelID,
		"TTLSeconds":     "31536000",
	}
	resp, status, err := c.performAPIRequest("CreateStreamKey", params)

	if err != nil {
		return "", err
	}

	if status != http.StatusOK {
		return "", errors.New(resp)
	}

	cskr := &createStreamKeyResp{}
	err = json.Unmarshal([]byte(resp), cskr)
	if err != nil {
		return "", err
	}

	return cskr.StreamKey.Name, nil
}

type createVideoChannelIDResp struct {
	VideoChannelID string `json:"Name"`
}

func (c *client) CreateVideoChannelID(channelID string) (string, error) {
	params := map[string]string{
		"Name": channelID,
	}
	resp, status, err := c.performAPIRequest("CreateVideoChannelID", params)

	if err != nil {
		return "", err
	}

	if status != http.StatusOK {
		return "", errors.New(resp)
	}

	cvci := &createVideoChannelIDResp{}
	err = json.Unmarshal([]byte(resp), cvci)
	if err != nil {
		return "", err
	}

	return cvci.VideoChannelID, nil
}

func (c *client) DeleteStreamKey(streamKey string) error {
	params := map[string]string{
		"Name": streamKey,
	}
	resp, status, err := c.performAPIRequest("DeleteStreamKey", params)

	if err != nil {
		return err
	}

	if status != http.StatusOK {
		return errors.New(resp)
	}

	return nil
}

func (c *client) performAPIRequest(url string, params map[string]string) (string, int, error) {
	now := time.Now()
	if c.lastReq.Add(rateLimit).After(now) {
		time.Sleep(rateLimit - now.Sub(c.lastReq))
	}

	req, err := http.NewRequest("POST", apiURL+url, nil)
	if err != nil {
		log.Fatalln("Failed to create request:", err)
	}

	jsonBody, err := json.Marshal(params)
	if err != nil {
		log.Fatalln("Failed to marshal JSON: ", err)
	}

	reader := bytes.NewReader(jsonBody)
	_, err = c.signer.Sign(req, reader, "svs", "us-west-2", time.Now())
	if err != nil {
		log.Fatalln("Failed to sign request:", err)
	}

	resp, err := c.httpClient.Do(req)
	c.lastReq = time.Now()
	if err != nil {
		return "", 0, err
	}

	body, _ := ioutil.ReadAll(resp.Body)

	if resp.StatusCode != http.StatusOK {
		eresp := &errResp{}
		_ = json.Unmarshal(body, eresp)
		return "", resp.StatusCode, fmt.Errorf(eresp.Message)
	}

	return string(body), resp.StatusCode, nil
}
