package auth

import (
	"context"
	"encoding/json"
	"net/http"

	rpc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/util"
	"github.com/sirupsen/logrus"
)

type key int

const (
	// KeyAmznLogin is the context key for the amazon sso login
	keyAmznLogin key = iota
)

// Auth interface defines the functions for the Auth handlers
type Auth interface {
	Handler(inner http.Handler) http.Handler
}

// GetUser returns the user from the supplied context
func GetUser(ctx context.Context) string {
	return ctx.Value(keyAmznLogin).(string)
}

// User is just a simple struct representing the logged in user
type User struct {
	Login       string                       `json:"login"`
	Permissions *rpc.GetModuleAccessResponse `json:"permissions"`
	TimeZones 	[]*util.TimeZone 			 `json:"timezones"`

}

// ForbiddenError is used in our temporary auth mechanism
type ForbiddenError struct {
	Error string `json:"error"`
}

// GetUserInfo is a tiny http handler that returns the login of the current user and their permissions
func GetUserInfo(carrotControl rpc.CarrotControl, logger logrus.FieldLogger) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		login := GetUser(r.Context())

		perms, err := carrotControl.GetModuleAccess(r.Context(), &rpc.GetModuleAccessRequest{})

		if err != nil {
			logger.WithError(err).Warn("Failed to get User Module Access")
		}

		timezones := util.ListTimeZones()

		w.WriteHeader(200)
		encoder := json.NewEncoder(w)
		err = encoder.Encode(User{Login: login, Permissions: perms, TimeZones: timezones })

		if err != nil {
			logger.WithError(err).Warn("Failed to encode user object")
		}
	})
}
