package channelviewer

import (
	"net/http"

	iocp "code.justin.tv/amzn/StarfruitIOCPTwirp"
	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/util"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type handler struct {
	mux    *goji.Mux
	cc     cc.CarrotControl
	logger logrus.FieldLogger
}

// NewChannelViewerHandler will forward channelviewer API requests to the carrot control service
func NewChannelViewerHandler(carrotControl cc.CarrotControl, logger logrus.FieldLogger) http.Handler {
	handler := &handler{
		mux:    goji.NewMux(),
		logger: logger,
		cc:     carrotControl,
	}

	handler.mux.HandleFunc(pat.New("/:customer_id/:region"), handler.ListChannels)
	handler.mux.HandleFunc(pat.New("/:customer_id/:region/:content_id"), handler.GetChannelDetail)

	return handler.mux
}

func (s *handler) ListChannels(writer http.ResponseWriter, request *http.Request) {
	customerID := pat.Param(request, "customer_id")
	region := pat.Param(request, "region")
	pageToken := request.URL.Query().Get("page_token")

	s.logger.Debugf("Calling ListStreamConfigs with customerID %v and region %v", customerID, region)
	resp, err := s.cc.ListStreamConfigs(request.Context(), &cc.ListStreamConfigsRequest{
		Region: region,
		Request: &iocp.ListStreamConfigsRequest{
			CustomerId: customerID,
			PageSize:   10,
			PageToken:  pageToken,
		},
	})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetChannelDetail(writer http.ResponseWriter, request *http.Request) {
	customerID := pat.Param(request, "customer_id")
	region := pat.Param(request, "region")
	contentID := pat.Param(request, "content_id")

	s.logger.Debugf("Calling GetStreamConfig with customerID %v, region %v and contentID %v", customerID, region, contentID)
	resp, err := s.cc.GetStreamConfig(request.Context(), &cc.GetStreamConfigRequest{
		Region: region,
		Request: &iocp.GetStreamConfigRequest{
			CustomerId:     customerID,
			StreamConfigId: contentID,
		},
	})

	util.HandleResponse(writer, resp, err)
}
