package reports

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	cac "code.justin.tv/event-engineering/carrot-analytics/control/rpc"
	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/api/auth"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/util"
	"github.com/golang/protobuf/ptypes"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type handler struct {
	mux    *goji.Mux
	cc     cc.CarrotControl
	logger logrus.FieldLogger
}

// NewReportsHandler will forward reports API requests to the carrot control service
func NewReportsHandler(carrotControl cc.CarrotControl, logger logrus.FieldLogger) http.Handler {
	handler := &handler{
		mux:    goji.NewMux(),
		logger: logger,
		cc:     carrotControl,
	}

	handler.mux.HandleFunc(pat.Post("/"), handler.SubmitQuery)
	handler.mux.HandleFunc(pat.Get("/"), handler.ListQueries)
	handler.mux.HandleFunc(pat.Get("/:report_id"), handler.GetReportDetail)

	return handler.mux
}

func (s *handler) ListQueries(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())

	s.logger.Debugf("Calling ListQueries with login %v", login)
	resp, err := s.cc.ListQueries(request.Context(), &cac.ListQueriesRequest{
		RequestedBy: login,
	})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetReportDetail(writer http.ResponseWriter, request *http.Request) {
	reportID := pat.Param(request, "report_id")

	s.logger.Debugf("Calling GetReportDetail with queryID %v", reportID)
	resp, err := s.cc.GetQueryResult(request.Context(), &cac.GetQueryResultRequest{
		QueryId: reportID,
	})

	util.HandleResponse(writer, resp, err)
}

type submitQueryRequest struct {
	QueryType          string `json:"query_type"`
	Start              string `json:"start"`
	End                string `json:"end"`
	PlaySessionID      string `json:"play_session_id"`
	BroadcastSessionID string `json:"broadcast_session_id"`
}

func (s *handler) SubmitQuery(writer http.ResponseWriter, request *http.Request) {

	var req submitQueryRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	login := auth.GetUser(request.Context())

	// We gotta convert the string to a go Time to a protobuf timestamp, the times we get from the front end are UTC
	startDate, err := time.Parse(time.RFC3339, req.Start)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	startTimestamp, err := ptypes.TimestampProto(startDate)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	endDate, err := time.Parse(time.RFC3339, req.End)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	endTimestamp, err := ptypes.TimestampProto(endDate)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	query := &cac.Query{
		StartDate: startTimestamp,
		EndDate:   endTimestamp,
	}

	fmt.Println(req.QueryType)
	label := ""

	switch req.QueryType {
	case "play_session_info":
		query.QueryParams = &cac.Query_PlaySessionInfoParams{
			PlaySessionInfoParams: &cac.PlaySessionInfoParameters{
				PlaySessionId: req.PlaySessionID,
			},
		}
		label = "Play Session Info: " + req.PlaySessionID
		break
	case "broadcast_session_info":
		query.QueryParams = &cac.Query_BroadcastInfoParams{
			BroadcastInfoParams: &cac.BroadcastInfoParameters{
				IngestSessionId: req.BroadcastSessionID,
			},
		}
		label = "Broadcast Session Info: " + req.BroadcastSessionID
		break
	default:
		util.HandleError(writer, http.StatusBadRequest, errors.New("Unable to determine query type"))
		return
	}

	s.logger.Debug("Calling EnqueueQuery")
	resp, err := s.cc.EnqueueQuery(request.Context(), &cac.EnqueueQueryRequest{
		Query:       query,
		RequestedBy: login,
		Label:       label,
	})

	util.HandleResponse(writer, resp, err)
}
