package videoanalysis

import (
	"encoding/json"
	"io/ioutil"
	"net/http"

	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	rtmpr "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/rpc"
	csa "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/rpc"
	auth "code.justin.tv/event-engineering/starfruit-support-portal/app/api/auth"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/util"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type handler struct {
	mux    *goji.Mux
	cc     cc.CarrotControl
	logger logrus.FieldLogger
}

// NewVideoAnalysisHandler will forward video analysis API requests to the carrot control service
func NewVideoAnalysisHandler(carrotControl cc.CarrotControl, logger logrus.FieldLogger) http.Handler {
	handler := &handler{
		mux:    goji.NewMux(),
		logger: logger,
		cc:     carrotControl,
	}

	// RTMP capture endpoint
	handler.mux.HandleFunc(pat.Get("/capture-endpoint"), handler.ListCaptureEndpoints)
	handler.mux.HandleFunc(pat.Get("/capture-endpoint-regions"), handler.GetAvailableCaptureEndpointRegions)
	handler.mux.HandleFunc(pat.Get("/capture-endpoint/:endpoint_id"), handler.GetCaptureEndpoint)
	handler.mux.HandleFunc(pat.Post("/capture-endpoint"), handler.CreateCaptureEndpoint)

	// RTMP Dump
	handler.mux.HandleFunc(pat.Get("/rtmp-dump"), handler.ListRTMPDumps)
	handler.mux.HandleFunc(pat.Get("/rtmp-dump/:rtmp_dump_id"), handler.GetRTMPDump)
	handler.mux.HandleFunc(pat.Post("/rtmp-dump"), handler.CreateRTMPDump)

	// We have to do these ones slightly differently because we expect the file key to contain slashes
	handler.mux.Handle(pat.Get("/captured-file/link/*"), http.StripPrefix("/captured-file/link/", &capturedFileDownloadHandler{h: handler}))
	handler.mux.Handle(pat.Get("/captured-file/*"), http.StripPrefix("/captured-file/", &capturedFileDetailHandler{h: handler}))

	return handler.mux
}

func (s *handler) ListCaptureEndpoints(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())

	s.logger.Debugf("Calling ListCaptureEndpoints with owner %v", login)
	resp, err := s.cc.ListCaptureEndpoints(request.Context(), &rtmpr.ListCaptureEndpointsRequest{
		Owner: login,
	})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetAvailableCaptureEndpointRegions(writer http.ResponseWriter, request *http.Request) {
	s.logger.Debug("Calling GetAvailableCaptureEndpointRegions")
	resp, err := s.cc.GetAvailableCaptureEndpointRegions(request.Context(), &rtmpr.GetAvailableCaptureEndpointRegionsRequest{})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetCaptureEndpoint(writer http.ResponseWriter, request *http.Request) {
	endpointID := pat.Param(request, "endpoint_id")

	s.logger.Debug("Calling GetCaptureEndpoint")
	resp, err := s.cc.GetCaptureEndpoint(request.Context(), &rtmpr.GetCaptureEndpointRequest{
		Id: endpointID,
	})

	util.HandleResponse(writer, resp, err)
}

type createCaptureEndpointRequest struct {
	Region             string `json:"region"`
	Name               string `json:"name"`
	MaxDurationSeconds uint64 `json:"max_duration_seconds"`
}

func (s *handler) CreateCaptureEndpoint(writer http.ResponseWriter, request *http.Request) {
	var req createCaptureEndpointRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	login := auth.GetUser(request.Context())

	s.logger.Debug("Calling CreateCaptureEndpoint")
	resp, err := s.cc.CreateCaptureEndpoint(request.Context(), &rtmpr.CreateCaptureEndpointRequest{
		Region:             req.Region,
		Name:               req.Name,
		MaxDurationSeconds: req.MaxDurationSeconds,
		Owner:              login,
	})

	util.HandleResponse(writer, resp, err)
}

type capturedFileDetailHandler struct {
	h *handler
}

func (s *capturedFileDetailHandler) ServeHTTP(writer http.ResponseWriter, request *http.Request) {
	s.h.logger.Debugf("Calling GetCapturedFileDetail with key %v", request.URL.Path)
	resp, err := s.h.cc.GetCapturedFileDetail(request.Context(), &csa.GetCapturedFileDetailRequest{
		Key: request.URL.Path,
	})

	util.HandleResponse(writer, resp, err)
}

type capturedFileDownloadHandler struct {
	h *handler
}

func (s *capturedFileDownloadHandler) ServeHTTP(writer http.ResponseWriter, request *http.Request) {
	s.h.logger.Debugf("Calling GetCapturedFileDownloadLink with key %v", request.URL.Path)
	resp, err := s.h.cc.GetCapturedFileDownloadLink(request.Context(), &csa.GetCapturedFileDownloadLinkRequest{
		Key: request.URL.Path,
	})

	util.HandleResponse(writer, resp, err)
}

type createRTMPDumpRequest struct {
	Region     string `json:"region"`
	Format     string `json:"format"`
	CustomerID string `json:"customer_id"`
	ChannelID  string `json:"channel_id"`
	Name       string `json:"name"`
}

func (s *handler) CreateRTMPDump(writer http.ResponseWriter, request *http.Request) {
	var req createRTMPDumpRequest
	bytes, err := ioutil.ReadAll(request.Body)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	err = json.Unmarshal(bytes, &req)
	if err != nil {
		util.HandleError(writer, http.StatusBadRequest, err)
		return
	}

	login := auth.GetUser(request.Context())

	s.logger.Debug("Calling CreateRTMPDump")
	resp, err := s.cc.CreateRTMPDump(request.Context(), &rtmpr.CreateRTMPDumpRequest{
		Region:          req.Region,
		Name:            req.Name,
		CustomerId:      req.CustomerID,
		ChannelId:       req.ChannelID,
		Format:          req.Format,
		DurationSeconds: 30, // TODO: Not sure what level of configurability we need yet for this so hard coding it for now
		Owner:           login,
	})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) ListRTMPDumps(writer http.ResponseWriter, request *http.Request) {
	login := auth.GetUser(request.Context())

	s.logger.Debugf("Calling ListRtmpDumps with owner %v", login)
	resp, err := s.cc.ListRTMPDumps(request.Context(), &rtmpr.ListRTMPDumpsRequest{
		Owner: login,
	})

	util.HandleResponse(writer, resp, err)
}

func (s *handler) GetRTMPDump(writer http.ResponseWriter, request *http.Request) {
	rtmpDumpID := pat.Param(request, "rtmp_dump_id")

	s.logger.Debug("Calling GetRTMPDump")
	resp, err := s.cc.GetRTMPDump(request.Context(), &rtmpr.GetRTMPDumpRequest{
		RtmpDumpId: rtmpDumpID,
	})

	util.HandleResponse(writer, resp, err)
}
