package util

import (
	"encoding/json"
	"errors"
	"log"
	"net/http"

	"code.justin.tv/event-engineering/starfruit-support-portal/app/api"
	"github.com/twitchtv/twirp"
)

// HandleError create an http response with the supplied status code, including the error supplied in err
func HandleError(writer http.ResponseWriter, statusCode int, err error) {
	if twErr, ok := err.(twirp.Error); ok {
		// Unwrap the twirp error if it's a twirp error
		err = errors.New(twErr.Msg())
	}

	err = sendResponse(writer, statusCode, api.Error{Message: err.Error()})

	if err != nil {
		log.Println(err)
	}
}

// HandleResponse will check for an error and attempt to map an http status code from a twirp error
// if there is no error then the resp is returned with a 200 status code
func HandleResponse(writer http.ResponseWriter, resp interface{}, err error) {
	var statusCode int
	var body interface{}

	if err != nil {
		statusCode = twirp.ServerHTTPStatusFromErrorCode(err.(twirp.Error).Code())
		body = api.Error{Message: err.(twirp.Error).Msg(), Code: err.(twirp.Error).Meta("Code")}
	} else {
		statusCode = 200
		body = resp
	}

	err = sendResponse(writer, statusCode, body)

	if err != nil {
		log.Println(err)
	}
}

// sendResponse will marshal the body to JSON and write it to the client
func sendResponse(writer http.ResponseWriter, statusCode int, body interface{}) error {
	writer.WriteHeader(statusCode)

	bytes, err := json.Marshal(body)
	if err != nil {
		return err
	}

	_, err = writer.Write(bytes)
	if err != nil {
		return err
	}

	return nil
}
