import cdk = require('@aws-cdk/core');
import lambda = require('@aws-cdk/aws-lambda');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam')

export interface PortalAppProps {
  vpcId: string
  trivEAARN: string
  carrotControlLambdaARN: string
  carrotControlClientID: string
}

export class PortalApp extends cdk.Construct {
  PortalLambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, props: PortalAppProps) {
    super(scope, id);

    let vpc = ec2.Vpc.fromLookup(this, 'VpcRef', {
      vpcId: props.vpcId,
    })

    this.PortalLambda = new lambda.Function(this, 'PortalLambda', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      handler: 'bin/portal-app/handler',
      code: lambda.Code.asset('../bin/portal-app.zip'),
      memorySize: 512,
      timeout: cdk.Duration.seconds(30),
      environment: {
        "carrotControlLambdaARN": props.carrotControlLambdaARN,
        "carrotControlClientID": props.carrotControlClientID,
      }
    })

    this.PortalLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: [
        "ssm:GetParameter",
        "ssm:GetParameters",
      ],
      resources: [
        cdk.Arn.format({
          partition: "aws",
          service: "ssm",
          account: cdk.Stack.of(this).account,
          region: cdk.Stack.of(this).region,
          resource: 'parameter',
          resourceName: 'carrot_csrfKey', // This created OOB and added manually to SSM
        }, cdk.Stack.of(this))
      ]
    }));

    this.PortalLambda.addToRolePolicy(new iam.PolicyStatement({
      effect: iam.Effect.ALLOW,
      actions: ["lambda:InvokeFunction"],
      resources: ["*"],
    }))

    this.PortalLambda.addPermission("PermissionForTrivEAToInvokeLambda", {
      principal: new iam.ServicePrincipal("apigateway.amazonaws.com"),
      sourceArn: props.trivEAARN,
      action: "lambda:InvokeFunction",
    })
  }
}
