package main

import (
	"bytes"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"strings"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	cacSvc "code.justin.tv/event-engineering/carrot-analytics/control/svc"
	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-control/pkg/svc"
	omnibar "code.justin.tv/event-engineering/carrot-omnibar/pkg/rpc"
	rtmpSvc "code.justin.tv/event-engineering/carrot-rtmp-recorder/pkg/svc"
	"code.justin.tv/event-engineering/carrot-stream-analysis/pkg/mmdb"
	csaSvc "code.justin.tv/event-engineering/carrot-stream-analysis/pkg/svc"
	cshSvc "code.justin.tv/event-engineering/carrot-system-health/pkg/svc"
	"code.justin.tv/video/amzncorp"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/sirupsen/logrus"
)

func generateLocalCCClient(sess *session.Session, iocpHost, iocpStage string, logger logrus.FieldLogger) cc.CarrotControl {
	// Carrot Stream Analysis
	streamlogEndpoint := "http://streamlog-query-prod.elasticbeanstalk.com"

	// Download the MMDB files using make mmdb
	// This command should be run from the root of the repository
	cityDbPath := "mmdb/GeoIP2-City.mmdb"
	ispDbPath := "mmdb/GeoIP2-ISP.mmdb"

	mmdbClient := mmdb.New(ispDbPath, cityDbPath, logger)

	flvAnalyserRecordingBucketName := "carrot-flv-analyser-recordings-stg"
	s3Client := s3.New(sess)

	csaServer := csaSvc.New(streamlogEndpoint, mmdbClient, s3Client, flvAnalyserRecordingBucketName, logger)

	// Carrot Analytics
	executorSQSQueueURL := "https://sqs.us-west-2.amazonaws.com/291428775132/carrot-analytics-prod-CarrotAnalyticsCarrotAnalyticsExecutorQueueC-Z1STG1KUQZH9"
	queriesTableName := "carrot-analytics-prod-CarrotAnalyticsCarrotAnalyticsQueries56AF22A9-12URBRWLGC5C0"
	resultsBucketName := "carrot-analytics-prod-carrotanalyticscarrotanalyt-1gt1xiojqbdnm"

	// Create the Service
	analyticsControlServer := cacSvc.New(sess, executorSQSQueueURL, queriesTableName, resultsBucketName, logger)

	// Omnibar
	// Just use the lambda function because I don't want this to need access to the weaver keys
	lambdaClient := lambda.New(sess)
	omnibarLambdaARN := "arn:aws:lambda:us-west-2:134277936734:function:carrot-omnibar-stg-OmnibarFunction2F151B77-M9L191IQKN42"
	lambdaTransport := twirpLambda.NewClient(lambdaClient, omnibarLambdaARN)

	omnibarClient := omnibar.NewCarrotOmnibarProtobufClient("https://would.you.like.a.carrot.twitch.tv", lambdaTransport)

	// IOCP
	iocpConfig := svc.IOCPConfig{
		Host:        iocpHost,
		Stage:       iocpStage,
		Credentials: sess.Config.Credentials, // Since IOCP allows invocation from anything in our account, our isengard credentials are sufficient here
	}

	omenHost := "http://omen.us-west-2.dev.gns.justin.tv/api/plyql/v1/query"
	druidBroker := "druid-broker-0-9-0-rc3.us-west-2.prod.gns.justin.tv:8082"
	omenAppName := "Carrot-ui-localdev"
	broodAPIURL := "http://alb.us-west-2.prod.brood.live-video.a2z.com"
	liveProxyAPIURL := "http://liveproxyapi-prod.elasticbeanstalk.com"
	originDCs := "sjc02,cmh01,lhr05,pdx05"
	cshServer, err := cshSvc.New(omenHost, druidBroker, omenAppName, broodAPIURL, liveProxyAPIURL, strings.Split(originDCs, ","), logger)

	if err != nil {
		panic(err)
	}

	// Carrot RTMP Recorder
	availableRegions := []string{"us-west-2"}
	digestionEndpoints := []string{
		"http://digestion.video.justin.tv",
		"http://digestion-cmh01.video.justin.tv",
		"http://direct.eu-west-2.prod.digestion.live-video.a2z.com",
	}
	endpointsTableName := "carrot-rtmp-recorder-api-stg-APIEndopintsTableD1E808A6-1RNAIDSYKX9OP"
	rtmpDumpsTableName := "carrot-rtmp-dumps"
	rtmpDumpQueueURL := "https://sqs.us-west-2.amazonaws.com/134277936734/rtmp-dump-queue"
	rtmpsDomain := "stg.support.contribute.live-video.net"
	ddbClient := dynamodb.New(sess)
	sqsClient := sqs.New(sess)

	rtmpRecorder := rtmpSvc.New(
		availableRegions,
		digestionEndpoints,
		endpointsTableName,
		rtmpDumpsTableName,
		rtmpDumpQueueURL,
		rtmpsDomain,
		ddbClient,
		s3Client,
		sqsClient,
		csaServer,
		logger,
	)

	return svc.New(csaServer, analyticsControlServer, cshServer, rtmpRecorder, omnibarClient, iocpConfig, nil, nil, svc.BindleLockConfig{}, logger)
}

func midwayCredentials(accountID, role string) (*credentials.Credentials, error) {
	midwayClient, err := midwayClient()
	if err != nil {
		return nil, err
	}

	return credentials.NewCredentials(&amzncorp.IsengardCredentials{
		AWSAccountID: accountID,
		IAMRoleName:  role,
		MidwayClient: midwayClient,
	}), nil

}

func midwayClient() (*http.Client, error) {
	cookieFile, err := readCookieFile()
	if err != nil {
		return nil, err
	}
	return amzncorp.MidwayClient(bytes.NewReader(cookieFile))
}

func readCookieFile() ([]byte, error) {
	home, err := os.UserHomeDir()
	if err != nil {
		return nil, err
	}

	f, err := os.Open(path.Join(home, ".midway/cookie"))
	if err != nil {
		return nil, err
	}

	defer f.Close()

	bs, err := ioutil.ReadAll(f)
	if err != nil {
		return nil, err
	}

	return bs, nil
}
