package main

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"sync"
	"time"

	ccSvc "code.justin.tv/event-engineering/carrot-control/pkg/svc"
	"code.justin.tv/jleroux/go-isengard2/service"
	"code.justin.tv/jleroux/go-isengard2/service/isengard"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/sirupsen/logrus"
	goji "goji.io"

	"code.justin.tv/event-engineering/starfruit-support-portal/app"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/api/auth"
)

// This is dev config only. Staging/Prod config is set up via enviromnent variables in CDK
var (
	carrotControlAccountID    = "134277936734" //starfruit-support+carrot-dev@amazon.com
	carrotControlIsengardRole = "CarrotDevelopers"
	carrotControlRegion       = "us-west-2"
	iocpHost                  = "iocp.live-video.a2z.com"
	iocpStage                 = "prod"
	carrotControlLambdaARN    = "arn:aws:lambda:us-west-2:134277936734:function:carrot-control-stg-CarrotControl89B6CA77-3BP4UEI0KLVD"
	carrotControlClientID     = "localdev" // Client ID is used to differentiate between clients in audits
)

type helloResp struct {
	User string `json:"user"`
}

func main() {
	var wg sync.WaitGroup

	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	// Create a new isengard client to establish who the user is and generate credentials necessary to contact services
	svc, err := service.NewIsengardService()
	if err != nil {
		fmt.Println("Failed to initiate isengard client (you probably need to run mwinit)")
		logger.Trace(err)
		os.Exit(1)
	}

	// Grab user identity with Hello
	r, err := svc.Hello(&isengard.HelloInput{})
	if err != nil {
		fmt.Println("Failed to establish developer identity (you probably need to run mwinit)")
		logger.Trace(err)
		os.Exit(1)
	}

	hr := &helloResp{}
	err = json.Unmarshal([]byte(*r.Message), hr)
	if err != nil {
		fmt.Printf("Failed to decode developer identity: %v\n", err)
		os.Exit(1)
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(carrotControlRegion),
		Credentials: credentials.NewCredentials(&isengardCredentialsProvider{
			svc:       svc,
			accountID: carrotControlAccountID,
			role:      carrotControlIsengardRole,
		}),
	})

	if err != nil {
		fmt.Printf("Failed to generate credentials for CarrotControl: %v\n", err)
		os.Exit(1)
	}

	// Create a dummy auth middleware for development using the identity that we've established via isengard
	rootMux := goji.NewMux()

	authHandler := auth.NewDummyAuth(hr.User)
	rootMux.Use(authHandler.Handler)

	// There are 2 options for working locally, you can either call the staging carrot control service remotely, or create the carrot control service locally
	// Unless you're messing with the services it's best to just call the remote lambda function
	carrotControl := generateLambdaCCClient(sess, carrotControlLambdaARN, carrotControlClientID)
	//carrotControl := generateLocalCCClient(sess, iocpHost, iocpStage, logger)

	// We want to pass the login through to the backend service, which we do with context values
	rootMux.Use(func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			ctx := r.Context()
			inner.ServeHTTP(w, r.WithContext(ccSvc.WithUserID(ctx, auth.GetUser(ctx))))
		})
	})

	app.New(logger, carrotControl, rootMux, "")

	fmt.Printf("Starting Dev Server as user %v\n", hr.User)
	wg.Add(1)
	go func() {
		defer wg.Done()

		err := http.ListenAndServe(":8337", rootMux)
		if err != nil {
			log.Println(err)
		}
	}()

	wg.Wait()
}

type isengardCredentialsProvider struct {
	svc        *isengard.Isengard
	accountID  string
	role       string
	expiration int64
}

func (icp *isengardCredentialsProvider) Retrieve() (credentials.Value, error) {
	output, err := icp.svc.GetAssumeRoleCredentials(&isengard.GetAssumeRoleCredentialsInput{
		AWSAccountID: aws.String(icp.accountID),
		IAMRoleName:  aws.String(icp.role),
	})

	if err != nil {
		return credentials.Value{}, err
	}

	result := &service.AssumeRoleResult{}
	err = json.Unmarshal([]byte(*output.AssumeRoleResult), result)
	if err != nil {
		return credentials.Value{}, err
	}

	icp.expiration = result.Credentials.Expiration

	return credentials.Value{
		AccessKeyID:     result.Credentials.AccessKeyID,
		SecretAccessKey: result.Credentials.SecretAccessKey,
		SessionToken:    result.Credentials.SessionToken,
		ProviderName:    "isengardCredentialsProvider",
	}, nil
}

func (icp *isengardCredentialsProvider) IsExpired() bool {
	return time.Now().Unix() >= icp.expiration
}
