package main

import (
	"fmt"
	"net/http"
	"os"

	twirpLambda "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	ccc "code.justin.tv/event-engineering/carrot-control/pkg/client"
	cc "code.justin.tv/event-engineering/carrot-control/pkg/rpc"
	"code.justin.tv/event-engineering/carrot-control/pkg/svc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/sirupsen/logrus"
	"goji.io"

	"code.justin.tv/event-engineering/starfruit-support-portal/app"
	"code.justin.tv/event-engineering/starfruit-support-portal/app/api/auth"
)

func main() {
	logger := logrus.New()
	logger.SetLevel(logrus.DebugLevel)

	carrotControlLambdaARN := os.Getenv("carrotControlLambdaARN")
	carrotControlClientID := os.Getenv("carrotControlClientID")

	// Create am AWS session
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	if err != nil {
		panic(fmt.Errorf("Error creating AWS session %v", err))
	}

	rootMux := goji.NewMux()

	// Auth all requests via midway
	authHandler := auth.NewMidwayAuth(logger)
	rootMux.Use(authHandler.Handler)

	// Set up carrot control client
	carrotControl := cc.NewCarrotControlProtobufClient("https://would.you.like.a.carrot.twitch.tv", ccc.New(carrotControlClientID, twirpLambda.NewClient(lambda.New(sess), carrotControlLambdaARN)))

	// We want to pass the login through to the backend service, which we do with context values
	rootMux.Use(func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			ctx := r.Context()
			inner.ServeHTTP(w, r.WithContext(svc.WithUserID(ctx, auth.GetUser(ctx))))
		})
	})

	// Grab csrf key from SSM
	ssmClient := ssm.New(sess)

	csrfKeyParam, err := ssmClient.GetParameter(&ssm.GetParameterInput{
		Name: aws.String("carrot_csrfKey"),
	})

	if err != nil {
		panic(err)
	}

	app.New(logger, carrotControl, rootMux, *csrfKeyParam.Parameter.Value)

	err = twirpLambda.ListenAndServe(":80", rootMux)
	if err != nil {
		// TODO logging and metrics
		panic(err)
	}
}
