import * as React from 'react';
import {AppContext, AppContextProps, ModalProps} from './appcontext';
import {TimeStamp} from "../../lib/api";
import {format as formatFn} from 'date-fns';

export interface Props {
}

export interface State {
    modalContent: React.ReactNode;
    props: AppContextProps;
    showModal: (modalContent: React.ReactNode, modalProps: ModalProps) => {};
    hideModal: () => {};
    timezoneOffset: number | undefined;
    setTimezone: (offset: number) => void;
    getTimezoneOffset: () => number | undefined;
    time: Time
}

interface Time {
    since: (timeStamp: TimeStamp) => string
    duration: (start: TimeStamp, end: TimeStamp, relativeSuffix: string) => string
    defaultFormat: (timeStamp: TimeStamp) => string
    format: (timeStamp: TimeStamp, dateFormat: string) => string
    modifyDateForTimezone: (inDate: Date) => Date
    timeStampToDate: (inTimeStamp: TimeStamp) => Date
}

const tzLocalStorageKey = 'carrot_timezone'

export class AppContextProvider extends React.Component<Props, State> {
    constructor(props: Props) {
        super(props);

        let selectedTimezone: number | undefined = undefined

        if (localStorage) {
            let lsTz = localStorage.getItem(tzLocalStorageKey)
            if (lsTz) {
                selectedTimezone = parseInt(lsTz)

                if (isNaN(selectedTimezone)) {
                    selectedTimezone = undefined
                }
            }
        }

        let time = {
            since: (timeStamp: TimeStamp): string => {
                return time.duration(timeStamp, {seconds: new Date().getTime() / 1000} as TimeStamp, ' ago');
            },
            duration: (start: TimeStamp, end: TimeStamp, relativeSuffix: string = '', relativePrefix: string = '', useEndDateAsRoot: boolean = false): string => {
                let startDate = time.modifyDateForTimezone(new Date(start.seconds * 1000))
                let endDate = time.modifyDateForTimezone(new Date(end.seconds * 1000))

                let secondsPast = (endDate.getTime() - startDate.getTime()) / 1000;

                if (secondsPast < 60) {
                    return relativePrefix + Math.round(secondsPast) + 's' + relativeSuffix;
                }
                if (secondsPast < 3600) {
                    return relativePrefix + Math.round(secondsPast / 60) + 'm' + relativeSuffix;
                }
                if (secondsPast <= 86400) {
                    return relativePrefix + Math.round(secondsPast / 3600) + 'h' + relativeSuffix;
                }

                if (secondsPast > 86400) {
                    let rootDate = startDate
                    if (useEndDateAsRoot) {
                        rootDate = endDate
                    }

                    let day = rootDate.getDate();
                    let monthMatches = rootDate.toDateString().match(/ [a-zA-Z]*/);
                    let month = '?';

                    if (monthMatches != null) {
                        month = monthMatches[0].replace(' ', '');
                    }
                    let year = startDate.getFullYear() == endDate.getFullYear() ? '' : ' ' + rootDate.getFullYear();
                    return day + ' ' + month + year;
                }

                return '?';
            },
            defaultFormat: (timeStamp: TimeStamp): string => {
                return time.format(timeStamp, 'yyyy-MM-dd HH:mm:ss');
            },
            format: (timeStamp: TimeStamp, dateFormat: string): string => {
                return formatFn(time.modifyDateForTimezone(new Date(timeStamp.seconds * 1000)), dateFormat);
            },
            modifyDateForTimezone: (inDate: Date): Date => {
                // Grab the desired timezone offset that the user has selected
                let tzo = this.getTimezoneOffset()

                // undefined here means the user just wants it to display in the timezone of their computer, so we don't need to do anything special
                if (tzo === undefined) {
                    return inDate
                }

                // Because JS Date takes in to account the timezone offset of the computer and we want something different
                // we effectively have to offset what JS wants to do for us before we create the new Date because JS is gonna do
                // it anyway
                let jsTZO = inDate.getTimezoneOffset() * -1

                // So we subtract what JS wants to do, then add what the user has asked us to do
                return new Date(inDate.getTime() - (jsTZO * 60000) + (tzo * 60000))
            },
            timeStampToDate: (inTimeStamp: TimeStamp): Date => {
                return time.modifyDateForTimezone(new Date(inTimeStamp.seconds * 1000))
            }
        }

        this.state = {
            modalContent: null,
            props: {
                modalProps: {},
            },
            showModal: this.showModal,
            hideModal: this.hideModal,
            timezoneOffset: selectedTimezone,
            setTimezone: this.setTimezone,
            getTimezoneOffset: this.getTimezoneOffset,
            time: time,
        };
    }

    public setTimezone = (offset: number | undefined) => {
        if (localStorage) {
            if (offset === undefined) {
                localStorage.removeItem(tzLocalStorageKey)
            } else {
                localStorage.setItem(tzLocalStorageKey, offset.toString())
            }
        }

        this.setState({
            timezoneOffset: offset,
        })
    }

    public getTimezoneOffset = (): number | undefined => {
        return this.state.timezoneOffset
    }

    public showModal = (modalContent: React.ReactNode, modalProps: ModalProps): any => {
        this.setState({
            modalContent,
            props: {
                ...this.props,
                modalProps,
            },
        });
    }

    public hideModal = (): any => {
        this.setState({
            modalContent: null,
            props: {
                ...this.props,
                modalProps: {},
            },
        });
    }

    public render() {
        return (
            <AppContext.Provider value={this.state}>
                {this.props.children}
            </AppContext.Provider>
        );
    }
}
