import {
  addDays,
  differenceInDays,
  differenceInMinutes,
  endOfDay,
  isEqual,
  isValid,
  subDays,
} from 'date-fns';

export interface Interval {
  start: Date;
  end: Date;
}

export function isEqualInterval(a: Interval, b: Interval) {
  return isEqual(a.start, b.start) && isEqual(a.end, b.end);
}

export function isSameLength(a: Interval, b: Interval) {
  return (
    a.end.getTime() - a.start.getTime() === b.end.getTime() - b.start.getTime()
  );
}

export function durationInMinutes(interval: Interval) {
  return differenceInMinutes(interval.end, interval.start);
}

export function toTimeParameters(interval: Interval) {
  return {
    start_time: interval.start.toJSON(),
    end_time: interval.end.toJSON(),
  };
}

export function lastNDays(n: number, now = new Date()): Interval {
  const endOfToday = endOfDay(now);

  return { start: subDays(endOfToday, n), end: endOfToday };
}

/**
 * Given an interval, create a new lookback interval that is shifted forward or
 * backward.
 */
export function shiftInterval(interval: Interval, direction: number): Interval {
  const dayDiff = differenceInDays(interval.end, interval.start);
  const shiftDays = (dayDiff + 1) * direction;

  return {
    start: addDays(interval.start, shiftDays),
    end: addDays(interval.end, shiftDays),
  };
}

export function intervalBeginsBefore(
  interval: Interval,
  otherInterval: Interval,
) {
  return otherInterval.start.getTime() < interval.start.getTime();
}

export function isIntervalValid(interval: Interval): boolean {
  return (
    isValid(interval.start) &&
    isValid(interval.end) &&
    interval.start.getTime() <= interval.end.getTime()
  );
}
