import {
  addDays,
  differenceInDays,
  isEqual,
  startOfDay,
  subDays,
} from 'date-fns';
import { Interval } from '../models/interval';

/**
 * Given a base interval, create a new interval that incorporates the values
 * from `patch`.
 *
 * We attempt to adjust the interval to preserve the same time range. If we
 * can't adjust the new interval to make it valid then we return the old
 * interval.
 */
export function updateInterval(
  oldInterval: Interval,
  patch: Partial<Interval>,
  now = new Date(),
): Interval {
  const newInterval = { ...oldInterval, ...patch };
  const newDiff = differenceInDays(newInterval.end, newInterval.start);

  // If the new interval is valid, just return the new interval
  if (newDiff >= 0) {
    return newInterval;
  }

  const oldDiff = differenceInDays(oldInterval.end, oldInterval.start);

  // If the user advances the start date past the end date, shift the end date
  // forward to preserve the same range of days. However, if the forward shift
  // would push this date beyond today, cap the end date to the start of today.
  if (isEqual(oldInterval.end, newInterval.end)) {
    const startOfToday = startOfDay(now);
    let end = addDays(newInterval.start, oldDiff);
    if (end > startOfToday) {
      end = startOfToday;
    }

    return { ...newInterval, end };
  }

  // If the user moves the start date back before the end date preserve the same
  // interval. This doesn't currently happen in practice because our date picker
  // prevents the user from selecting end dates before the start date.
  if (isEqual(oldInterval.start, newInterval.start)) {
    return { ...newInterval, start: subDays(newInterval.end, oldDiff) };
  }

  // The new interval is not valid and we can't make any smart adjustments to the
  // range because both the start date and end date are new.
  return oldInterval;
}
