import { startOfDay } from 'date-fns';
import * as Pikaday from 'pikaday';
import { PikadayOptions } from 'pikaday';
import * as React from 'react';
import './styles.scss';

interface Props {
  onChange: (date: Date) => void;
  startRange: Date;
  endRange: Date;
  minDate?: Date;
  maxDate: Date;
  date: Date;
}

/**
 * This DatePicker is a thin wrapper around the Pikaday library.
 *
 * https://github.com/dbushell/Pikaday
 *
 * This component should only provide a sensible component interface to Pikaday
 * and should not perform any validation or contain application logic. There
 * are no automated tests for this component since we do not own the interface
 * to Pikaday and cannot be confident that this component is working without
 * testing its integration against the full library in a real web browser.
 */
export class DatePicker extends React.Component<Props> {
  private picker: Pikaday;

  public componentDidUpdate() {
    // It's convenient for us to accept dates that are the end of a day, since
    // this fits with the model of how data is queried. For instance, selecting
    // Feb 1 in each picker should query for one whole day.
    //
    // However, Pikaday is picky about the extact time needed to set the date of
    // the picker: each date must be the beginning of the day.
    this.picker.setStartRange(startOfDay(this.props.startRange));
    this.picker.setEndRange(startOfDay(this.props.endRange));
    if (this.props.minDate) {
      this.picker.setMinDate(startOfDay(this.props.minDate));
    }
    this.picker.setMaxDate(startOfDay(this.props.maxDate));
    this.picker.setDate(
      startOfDay(this.props.date),
      true /* suppress onSelect */,
    );
  }

  public render() {
    return <div ref={this.setContainerRef} />;
  }

  private setContainerRef = (container: HTMLDivElement | null) => {
    if (container === null) {
      this.picker.destroy();
    } else {
      this.picker = new Pikaday({
        onSelect: this.props.onChange,
        theme: 'inline',
        minDate: this.props.minDate && startOfDay(this.props.minDate),
        maxDate: startOfDay(this.props.maxDate),
        defaultDate: startOfDay(this.props.date),
        startRange: startOfDay(this.props.startRange),
        endRange: startOfDay(this.props.endRange),
        setDefaultDate: true,
        // One day.... we'll do this properly
        i18n: {
          previousMonth: 'Previous Month',
          nextMonth: 'Next Month',
          months: [
            'January',
            'February',
            'March',
            'April',
            'May',
            'June',
            'July',
            'August',
            'September',
            'October',
            'November',
            'December',
          ],
          weekdays: [
            'Monday',
            'Tuesday',
            'Wednedsay',
            'Thursday',
            'Friday',
            'Saturday',
            'Sunday',
          ],
          weekdaysShort: [
            'Mon',
            'Tue',
            'Wed',
            'Thu',
            'Fri',
            'Sat',
            'Sun',
          ],
        },
      } as PikadayOptions);

      container.appendChild(this.picker.el);
    }
  }
}
