import {
  endOfDay,
  endOfMonth,
  startOfDay,
  startOfMonth,
  subDays,
  subMonths,
} from 'date-fns';
import * as React from 'react';
import { Interval, isEqualInterval } from 'common/models/interval';

import {
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  Layout,
  StyledLayout,
  TextAlign,
} from 'twitch-core-ui';

interface Props {
  interval: Interval;
  onChange: (interval: Interval) => void;
  now?: Date;
}

export enum PresetID {
  Custom = 'custom',
  Default = 'default',
  Last7Days = 'last_7_days',
  Last30Days = 'last_30_days',
  CurrentMonth = 'current_month',
  oneMonthAgo = 't_minus_1_month',
  twoMonthsAgo = 't_minus_2_month',
}

export interface PresetInterval extends Interval {
  id: PresetID;
}

export class DateRangePickerPresets extends React.Component<Props, {}> {
  private presets: Array<{ message: string; interval: PresetInterval }>;

  constructor(props: Props) {
    super(props);

    const now = props.now || new Date();
    const endOfToday = endOfDay(now);
    const startOfToday = startOfDay(now);
    const startOfThisMonth = startOfMonth(now);

    this.presets = [
      {
        message: 'Last 7 Days',
        interval: {
          start: subDays(startOfToday, 6),
          end: endOfToday,
          id: PresetID.Last7Days,
        },
      },
      {
        message: 'Last 30 Days',
        interval: {
          start: subDays(startOfToday, 29),
          end: endOfToday,
          id: PresetID.Last30Days,
        },
      },
      {
        message: 'This month so far',
        interval: {
          start: startOfThisMonth,
          end: endOfToday,
          id: PresetID.CurrentMonth,
        },
      },
      {
        message: 'Last Month',
        interval: {
          start: startOfMonth(subMonths(now, 1)),
          end: endOfMonth(subMonths(now, 1)),
          id: PresetID.oneMonthAgo,
        },
      },
      {
        message: '2 Months Ago',
        interval: {
          start: startOfMonth(subMonths(now, 2)),
          end: endOfMonth(subMonths(now, 2)),
          id: PresetID.twoMonthsAgo,
        },
      },
    ];
  }

  public render() {
    return (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
        {this.presets.map((preset, i) => (
          <StyledLayout color={Color.Link} margin={{ bottom: 0.5 }} key={i}>
            <Interactable
              selected={isEqualInterval(this.props.interval, preset.interval)}
              onClick={this.handleClickFor(preset.interval)}
              borderRadius={BorderRadius.Small}
            >
              <StyledLayout padding={1} textAlign={TextAlign.Center}>
                <CoreText
                  fontSize={FontSize.Size5}
                >
                  {preset.message}
                </CoreText>
              </StyledLayout>
            </Interactable>
          </StyledLayout>
        ))}
      </Layout>
    );
  }

  private handleClickFor(interval: Interval) {
    return () => this.props.onChange(interval);
  }
}
