import { endOfDay } from 'date-fns';
import * as React from 'react';
import { Interval } from 'common/models/interval';
import { updateInterval } from 'common/utils/interval';
import {
  Color,
  CoreText,
  Display,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  StyledLayout,
  TextAlign,
} from 'twitch-core-ui';
import { DatePicker } from '../datepicker';
import {
  DateRangePickerPresets,
  PresetID,
  PresetInterval,
} from '../daterangepicker-presets';

interface Props {
  interval: Interval;
  onCommit: (interval: PresetInterval) => void;
}

interface State {
  interval: Interval;
}

export const COMMIT_BUTTON_SELECTOR = { 'data-test-selector': 'commit-button' };
export const CANCEL_BUTTON_SELECTOR = { 'data-test-selector': 'cancel-button' };

/**
 * This component renders other components that allow users to select a time
 * range to view. Rules for which days to enable in the date pickers are
 * configured here.
 *
 * `this.state` temporarily stores the interval as the user interacts with the
 * selection UI. Changes to the dates are passed through through the
 * `updateInterval` policy function which may adjust the interval before the
 * state changes. The user then commits or discards the state when they are
 * done selecting.
 */
export class DateRangePicker extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);
    this.state = { interval: this.props.interval };
  }

  public render() {
    const now = new Date();

    return (
      <div>
        <Layout
          display={Display.Flex}
          flexWrap={FlexWrap.NoWrap}
          margin={{ bottom: 2 }}
          justifyContent={JustifyContent.Center}
        >
          <Layout padding={{ right: 2 }} textAlign={TextAlign.Center}>
            <CoreText color={Color.Alt2} fontSize={FontSize.Size4}>
              Start date
            </CoreText>
            <DatePicker
              date={this.state.interval.start}
              startRange={this.state.interval.start}
              endRange={this.state.interval.end}
              maxDate={now}
              onChange={this.handleStartChange}
            />
          </Layout>
          <Layout padding={{ right: 2 }} textAlign={TextAlign.Center}>
            <CoreText color={Color.Alt2} fontSize={FontSize.Size4}>
              End date
            </CoreText>
            <DatePicker
              date={this.state.interval.end}
              startRange={this.state.interval.start}
              endRange={this.state.interval.end}
              minDate={this.state.interval.start}
              maxDate={now}
              onChange={this.handleEndChange}
            />
          </Layout>
          <StyledLayout padding={{ top: 0.5, left: 2 }} borderLeft>
            <DateRangePickerPresets
              interval={this.state.interval}
              onChange={this.handlePresetChange}
            />
          </StyledLayout>
        </Layout>
      </div>
    );
  }

  private handlePresetChange = (interval: PresetInterval) => {
    this.setState({ interval });
    return this.props.onCommit(interval);
  }

  private handleStartChange = (start: Date) => {
    const interval = updateInterval(this.state.interval, { start });
    this.setState({ interval });
    this.props.onCommit({ ...this.state.interval, id: PresetID.Custom });
  }

  private handleEndChange = (end: Date) => {
    const interval = updateInterval(this.state.interval, {
      end: endOfDay(end),
    });
    this.setState({ interval });
    this.props.onCommit({ ...this.state.interval, id: PresetID.Custom });
  }
}
