import * as React from 'react';
import { AppContext } from 'common/appcontext';
import { APIError } from 'lib/api';
import { formatBits } from 'common/utils/bytes';
import { CoreText, Layout, LoadingSpinner, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { getLimitBreaches } from './api';
import { SECEvent } from 'common/sec/model'
import { Paging, Position } from 'components/paging'

interface Props {
  customer_id: string;
  region: string;
  session_id: string;
}

interface State {
  limit_breaches?: SECEvent[];
  processing: boolean;
  next_page_token?: string;
  page_stack: string[]
}

export class LimitBreachList extends React.Component<Props, State> {
  public state: State = {
    processing: true,
    page_stack: [],
  };

  public render() {
    let content = <></>;

    if (this.state && this.state.limit_breaches && this.state.limit_breaches.length > 0) {
      content = (
        <TableBody>
          {this.state.limit_breaches.map((event, index) => (
            <TableRow key={index}>
              <TableCell>{this.context.time.defaultFormat(event.event_time)}</TableCell>
              <TableCell>{event.event_name}</TableCell>
              <TableCell>{LimitBreachList.formatDetail(event)}</TableCell>
          </TableRow>
          ))}
        </TableBody>
      );
    }

    return (
      <Layout padding={{bottom:3}}>
        <Layout padding={{ bottom: 3 }}>
          <CoreText type={TextType.H3}>Latest Limit Breaches</CoreText>
        </Layout>
        <Layout>
          <Paging loading={this.state.processing} position={Position.Before} prevPageExists={this.state.page_stack.length > 0} nextPageExists={!!this.state.next_page_token} triggerNextPage={() => this.nextPage()} triggerPrevPage={() => this.prevPage()}>
            <Table>
              <TableHeader>
                <TableHeading label="Time" />
                <TableHeading label="Event" />
                <TableHeading label="Detail" />
              </TableHeader>
              {content}
            </Table>
          </Paging>
        </Layout>
      </Layout>
    );
  }

  public componentWillMount() {
    this.getLimitBreachPage('');
  }

  private nextPage() {
    if (!this.state) {
      return
    }

    let ps = this.state.page_stack;
    ps.push(this.state.next_page_token)

    this.setState({
      page_stack: ps,
    })

    this.getLimitBreachPage(this.state.next_page_token);
  }

  private prevPage() {
    let ps = this.state.page_stack;
    ps.pop()

    this.setState({
      page_stack: ps,
    })

    let page_token = '';

    if (ps.length > 0){
      page_token = ps[ps.length - 1]
    }

    this.getLimitBreachPage(page_token);
  }

  public static formatDetail(event: SECEvent) :string {
    let data = JSON.parse(event.extra)
    switch (event.event_name) {
      case 'CCV_LIMIT_EXCEEDED':
        return `Concurrent viewer limit of ${data.limit_threshold} was exceeded by ${data.exceeded_by}`;
      case 'BITRATE_LIMIT_EXCEEDED':
        return `Bitrate limit of ${formatBits(data.limit_threshold)} was exceeded by ${formatBits(data.exceeded_by)}`;
      case 'PIXEL_LIMIT_EXCEEDED':
        return `Resolution limit of ${data.limit_threshold} total pixels was exceeded by ${data.exceeded_by}`;
      case 'CCB_LIMIT_EXCEEDED':
        return `Concurrent broadcasters limit of ${data.limit_threshold} was exceeded by ${data.exceeded_by || 0/*ING-8724*/}`;
    }
    return 'Unknown Event Type';
  }

  private async getLimitBreachPage(page_token :string) {
    this.setState({
      processing: true,
    });

    await getLimitBreaches(this.props.customer_id, this.props.region, this.props.session_id, page_token).then((limitBreachResponse) => {
      this.setState({
        limit_breaches: limitBreachResponse.limit_breach_events,
        next_page_token: limitBreachResponse.next_page_token,
        processing: false,
      });
    }, (reason: APIError) => {
      this.setState({
        processing: false,
      });

      if (reason.status == 404){
        return
      }

      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Getting Limit Breach List',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

LimitBreachList.contextType = AppContext;
