import * as classNames from 'classnames';
import * as React from 'react';
import { ClickOutDetector } from 'components/click-out-detector';
import {
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  Interactable,
  Layout,
  Overflow,
  Placeholder,
  Position,
  StyledLayout,
  SVG,
  SVGAsset, TableCell,
  ZIndex,
  Tooltip, TooltipDirection
} from 'twitch-core-ui';
import { UserInfo } from '../../../App';
import './timezone-selector.sass';
import {AppContext} from "../../../common/appcontext";

interface Props {
  userInfo: UserInfo | null;
}

interface State {
  dropdownOpen: boolean;
  currentValue: number | undefined
}

interface OffsetProps {
  offsetMinutes: number
}

class TZOffset extends React.Component<OffsetProps, { }> {
  constructor(props: OffsetProps, state: { }) {
    super(props, state);
  }

  public render() {
    return <>UTC {this.formatOffset(this.props.offsetMinutes)}</>
  }

  private formatOffset = (offset: number) :string => {
    // glhf
    return (offset < 0 ? '-' : '+') + ("00" + (Math.floor(Math.abs(offset) / 60))).slice(-2) + ':' + ("00" + (Math.abs(offset) - Math.floor(Math.abs(offset) / 60) * 60)).slice(-2)
  }
}

export class TimezoneSelector extends React.Component<Props, State> {
  constructor(props: Props, state: State) {
    super(props, state);

    this.state = {
      dropdownOpen: false,
      currentValue: this.context.getTimezoneOffset(),
    };
  }

  public render() {
    const dropdownClassname = classNames.default('top-nav_timezone-dropdown', { 'top-nav_timezone-dropdown--toggled': this.state.dropdownOpen });

    let timezoneDropDown = (
      <Layout flexGrow={0} flexShrink={0} display={Display.Flex} alignItems={AlignItems.Start}>
        <Layout padding={{ left: 0.5 }}>
          <Placeholder overlay width={200} height={14} />
          <Placeholder overlay lineCount={1} />
        </Layout>
      </Layout>
    );

    if (this.props.userInfo) {

      let tzi = <></>
      let className = ''

      if (this.state.currentValue !== undefined && new Date().getTimezoneOffset() * -1 != this.state.currentValue) {
        className = 'tz_selected'

        tzi = (
            <Tooltip label={"Don't forget! You've selected a specific time zone that is different to your local time zone"} direction={TooltipDirection.Right} width={200}>
              <Layout padding={{left: 1}} display={Display.InlineBlock} className={'blink_me'}>🥕</Layout>
            </Tooltip>
        )
      }

      timezoneDropDown = (
        <Layout className={className} flexGrow={0} flexShrink={0} display={Display.Flex} alignItems={AlignItems.Start}>
          <Interactable
            onClick={this.onUserMenuClick}
            blurAfterClick
          >
            <Layout
              display={Display.Flex}
              flexGrow={1}
              alignItems={AlignItems.Stretch}
              fullHeight
              padding={1}
            >
              <Layout padding={{ left: 0.5 }} className="top-nav_user-info">
                <StyledLayout>
                  <CoreText>Timezone</CoreText>
                </StyledLayout>
                <TZOffset offsetMinutes={this.state.currentValue === undefined ? new Date().getTimezoneOffset() * -1 : this.state.currentValue}/>{tzi}
              </Layout>
              <StyledLayout
                display={Display.Flex}
                flexShrink={0}
                color={Color.Base}
              >
                <SVG asset={SVGAsset.GlyphArrDown} />
              </StyledLayout>
            </Layout>
          </Interactable>
        </Layout>
      );
    }

    let list = <></>
    if (this.props.userInfo) {
      list = (
        <Layout margin={{ x: 2, bottom: 1 }}>
          <Interactable onClick={() => this.onClick(undefined)}>
            <Layout>Auto: <TZOffset offsetMinutes={new Date().getTimezoneOffset() * -1} /></Layout>
          </Interactable>
          <StyledLayout borderBottom margin={{ x: 2, y: 1 }} />
          {this.props.userInfo.timezones.map((tz, index) => (
            <Interactable key={index} onClick={() => this.onClick(tz.offset_minutes)}>
              <TZOffset offsetMinutes={tz.offset_minutes}/>
            </Interactable>
          ))}
        </Layout>
      )
    }

    return (
      <Layout position={Position.Relative} display={Display.Flex} flexGrow={0} alignItems={AlignItems.Stretch} fullHeight className="tw-root--theme-dark">
        <ClickOutDetector onClickOut={this.onUserMenuClickOut} >
          {timezoneDropDown}
          <StyledLayout
            overflow={Overflow.Hidden}
            position={Position.Absolute}
            className={dropdownClassname}
            background={Background.Alt}
            color={Color.Base}
            attachRight
            fullWidth
            zIndex={ZIndex.Below}
            padding={{ top: 1, bottom: 1 }}
            elevation={3}
          >
            {list}
          </StyledLayout>
        </ClickOutDetector>
      </Layout>

    );
  }

  private onUserMenuClickOut = () => {
    this.setState({ dropdownOpen: false });
  }

  private onUserMenuClick = () => {
    if (!this.props.userInfo) {
      return;
    }
    this.setState({ dropdownOpen: !this.state.dropdownOpen });
  }

  private onClick = (offsetMinutes: number | undefined) => {
    this.context.setTimezone(offsetMinutes);
    this.setState({
      currentValue: offsetMinutes,
      dropdownOpen: false,
    })
  }
}

TimezoneSelector.contextType = AppContext;
