import * as classNames from 'classnames';
import * as React from 'react';
import { ClickOutDetector } from 'components/click-out-detector';
import { ThemeSwitcher } from 'components/theme-switcher';
import { AlignItems, Background, Color, CoreText, Display, Interactable, Layout, Overflow, Placeholder, Position, StyledLayout, SVG, SVGAsset, ZIndex } from 'twitch-core-ui';
import { UserInfo } from '../../../App';
import './user-menu-dropdown.css';

interface Props {
  userInfo: UserInfo | null;
  logout: () => void;
}

interface State {
  dropdownOpen: boolean;
}

export class UserMenuDropdown extends React.Component<Props, State> {
  constructor(props: Props, state: State) {
    super(props, state);

    this.state = {
      dropdownOpen: false,
    };
  }

  public render() {
    const dropdownClassname = classNames.default('top-nav_user-info-dropdown', { 'top-nav_user-info-dropdown--toggled': this.state.dropdownOpen });

    let placeholder = (
      <Layout flexGrow={0} flexShrink={0} display={Display.Flex} alignItems={AlignItems.Start}>
        <Layout padding={{ left: 0.5 }}>
          <Placeholder overlay width={200} height={14} />
          <Placeholder overlay lineCount={1} />
        </Layout>
      </Layout>
    );

    let userDropDown = placeholder;

    if (this.props.userInfo) {
      userDropDown = (
        <Layout flexGrow={0} flexShrink={0} display={Display.Flex} alignItems={AlignItems.Start}>
          <Interactable
            onClick={this.onUserMenuClick}
            blurAfterClick
          >
            <Layout
              display={Display.Flex}
              flexGrow={1}
              alignItems={AlignItems.Stretch}
              fullHeight
              padding={1}
            >
              <Layout padding={{ left: 0.5 }} className="top-nav_user-info">
                <StyledLayout>
                  <CoreText>{this.props.userInfo.login}</CoreText>
                </StyledLayout>
                <Layout>
                  <CoreText>LDAP: {this.props.userInfo.login}</CoreText>
                </Layout>
              </Layout>
              <StyledLayout
                display={Display.Flex}
                flexShrink={0}
                color={Color.Base}
              >
                <SVG asset={SVGAsset.GlyphArrDown} />
              </StyledLayout>
            </Layout>
          </Interactable>
        </Layout>
      );
    }

    return (
      <Layout position={Position.Relative} display={Display.Flex} flexGrow={0} alignItems={AlignItems.Stretch} fullHeight className="tw-root--theme-dark">
        <ClickOutDetector onClickOut={this.onUserMenuClickOut} >
          {userDropDown}
          <StyledLayout
            overflow={Overflow.Hidden}
            position={Position.Absolute}
            className={dropdownClassname}
            background={Background.Alt}
            color={Color.Base}
            attachRight
            fullWidth
            zIndex={ZIndex.Below}
            padding={{ top: 1, bottom: 1 }}
            elevation={3}
          >
            <Layout margin={{ x: 2, bottom: 1 }}>
              <ThemeSwitcher
                availableThemes={[
                  { label: 'Light', className: 'tw-root--theme-light', default: true },
                  { label: 'Dark', className: 'tw-root--theme-dark' },
                ]}
              />
            </Layout>
            <StyledLayout borderBottom margin={{ x: 2, y: 1 }} />
            <Interactable
              onClick={this.props.logout}
            >
              <StyledLayout
                display={Display.Flex}
                alignItems={AlignItems.Center}
                padding={{ x: 2, y: 0.5 }}
              >
                <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ right: 1 }}>
                  <SVG asset={SVGAsset.NavLogout} height={20} width={20} />
                </Layout>
                <CoreText>Logout</CoreText>
              </StyledLayout>
            </Interactable>
          </StyledLayout>
        </ClickOutDetector>
      </Layout>

    );
  }

  private onUserMenuClickOut = () => {
    this.setState({ dropdownOpen: false });
  }

  private onUserMenuClick = () => {
    if (!this.props.userInfo) {
      return;
    }
    this.setState({ dropdownOpen: !this.state.dropdownOpen });
  }
}
