import { API, TimeStamp, Duration } from 'lib/api';

export enum ResponseType {
  GenericMessage = 0,
  TwitchManifestURL = 1,
  IVSManifestURL = 10,
  TwitchWeaverURL = 20,
  IVSWeaverURL = 30,
  TwitchSegmentURL = 40,
  IVSSegmentURL = 50,
  TwitchStreamKey = 60,
  IVSStreamKey = 70,
  IVSChannelARN = 80,
  AVCProfile = 90,
}

export interface OmnibarResponse {
  type: ResponseType;
  generic_message: string;
  ivs_customer_id: string;
  ivs_region: string;
  ivs_channel_id: string;
  ivs_channel_region: string;
  twitch_channel: string;
  twitch_channel_id: string;
  edge_node: string;
  edge_cluster: string;
  abs_type: string;
  mux_type: string;
  weaver_cluster: string;
  ingest_pop: string;
  weaver_payload_json: string;
  avc_profile_level: string;
  avc_profile_name: string;
}

export interface OmnibarError {
  code: number
  message: string
}

export enum Protocol {
  UNKNOWN = 0,
  HTTP = 1,
  HTTPS = 2,
}

export interface UserEdgeRequest {
  protocol: Protocol;
  node: string;
  expiration: TimeStamp;
  synthetic: boolean;
  event_node: string;
}

export interface PlayerDetails {
  platform: string;
  player: string;
}

export enum ContentType {
  UNKNOWN = 0,
  TWITCH = 1,
  LVS = 2,
}

export interface AnalyticsData {
  channel_name: string;
  customer_id: string;
  content_id: string;
  content_type: ContentType;
  channel_id: string;
}

export interface NodeRequestStream {
  id: number;
  channel: string;
  low_latency: boolean;
  origin: string;
}

export interface UserClassification {
  ip: string;
  entry_pop: string;
  country: string;
  cidr24: string;
  device_id: string;
}

export interface NodeRequestOptions {
  force_cluster: string;
  https_required: boolean;
  warp_requested: boolean;
}

export interface CustomerInfo {
  customer_id: string;
  content_id: string;
}

export interface NodeRequest {
  stream: NodeRequestStream;
  user: UserClassification;
  options: NodeRequestOptions;
  customer: CustomerInfo;
  request_id: string;
  session_id: string;
  current_service_node: ServiceNode;
}

export enum NetworkEnvironment {
  UNKNOWN = 0,
  ONNET = 1,
  OFFNET = 2,
  CDN = 3,
}

export enum DeliveryProtocol {
  PROTOCOL_UNKNOWN = 0,
  HLS = 1,
  WARP = 2,
}

export interface ServiceNode {
  name: string;
  cluster: string;
  network_environment: NetworkEnvironment;
  delivery_protocol: DeliveryProtocol;
}

export interface PlaybackSession {
  id: number;
  ppid: string;
  authenticated: boolean;
  user_id: string;
  player_details: PlayerDetails;
  analytics_data: AnalyticsData;
  assignment_request: NodeRequest;
  hls_node: ServiceNode;
  warp_node: ServiceNode;
}

export enum EncryptionScheme {
  NONE = 0,
  CENC = 1,
  CBCS = 2,
}

export interface Stream {
  broadcast_id: string;
  usher_stream_id: number;
  quality: string;
  origin_dc: string;
  encryption_scheme: EncryptionScheme;
  origin_rendition_path: string;
  abs_available: boolean;
  channel_arn: string;
  video_height: number;
}

export interface Beacon {
  host: string;
  payload: string;
}

export interface PlaylistRequest {
  window_duration: Duration;
  future_segments_enabled: boolean;
  edge_reassignments_enabled: boolean;
}

export interface SegmentRequest {
  uri: string;
  advertisement: boolean;
  beacon: Beacon;
  beacon_ids: string[];
  sequence_no: number;
  billing_duration: Duration;
}

export enum AdvertisingType {
  UNKNOWN = 0,
  DISABLED = 1,
  CLIENT_STITCHED = 2,
  SERVER_STITCHED = 3,
}

export interface Advertising {
  metadata: string;
  preroll_type: AdvertisingType;
  midroll_type: AdvertisingType;
  postroll_type: AdvertisingType;
  experiment_midroll_timeout: Duration;
}

export interface WeaverRequestInfo {
  request: UserEdgeRequest
  session: PlaybackSession;
  stream: Stream;
  playlist: PlaylistRequest;
  segment: SegmentRequest;
  advertising: Advertising;
}

export async function whatIs(thing: string): Promise<OmnibarResponse> {
  const path = `/api/omnibar/whatis?thisthing=` + encodeURI(thing);

  const response = await API.get<OmnibarResponse>(path.trim(), {});
  if (response.error) {
    let errResp: OmnibarError = {
      code: response.status,
      message: `Sorry, I couldn't figure out what this is :(`
    }
    return Promise.reject(errResp);
  }

  if (response.body === undefined) {
    let errResp: OmnibarError = {
      code: 400,
      message: `Sorry, I couldn't figure out what this is :(`
    }
    return Promise.reject(errResp);
  }

  // enums are weird
  if (!response.body.type) {
    response.body.type = 0;
  }

  return response.body;
}
