import * as React from 'react';
import * as CopyToClipboard from 'react-copy-to-clipboard';
import { Link } from 'react-router-dom';
import { Color, CoreInteractive, CoreText, FontSize, Layout, SVG, SVGAsset, SVGType, TextType, Tooltip, TooltipDirection } from 'twitch-core-ui';

export enum DataClassification {
  Internal,
  CustomerOnly,
  External,
}

export interface PanelItemWarning {
  description: string;
}

interface Props {
  name: string;
  value: string | number;
  hideCopy?: boolean;
  linkTo?: string;
  dataClassification: DataClassification;
  outOfSpec?: React.ReactElement;
  onClick?: () => void;
  panelItemWarning?: PanelItemWarning | undefined;
}

export class PanelItem extends React.Component<Props, {}> {
  constructor(props: Props) {
    super(props);
  }

  public render() {
    if (this.props.value !== undefined) {
      let color = Color.Error;

      if (this.props.dataClassification == DataClassification.CustomerOnly) {
        color = Color.Link;
      }

      if (this.props.dataClassification == DataClassification.External) {
        color = Color.Base;
      }

      let warning = (<></>);
      if (this.props.panelItemWarning) {
        warning = (
          <Tooltip label={this.props.panelItemWarning.description} direction={TooltipDirection.Right} width={200}>
            <SVG type={SVGType.Warn} width={16} height={16} asset={SVGAsset.Warning} />&nbsp;
          </Tooltip>
        );
      }

      let copy = (<></>);
      if (!this.props.hideCopy) {
        copy = (
          <CoreText type={TextType.Span} color={Color.Alt}>
            <CopyToClipboard text={this.props.value.toString()}>
              <CoreInteractive blurAfterClick={true}>
                <SVG type={SVGType.Alt2} width={16} height={16} asset={SVGAsset.Copy} />
              </CoreInteractive>
            </CopyToClipboard>
          </CoreText>
        );
      }

      let value = (
        <CoreText type={TextType.Span} fontSize={FontSize.Size4}>{this.props.value}</CoreText>
      );

      if (this.props.linkTo) {
        value = (
          <Link to={this.props.linkTo}><CoreText type={TextType.Span} fontSize={FontSize.Size4}>{this.props.value}</CoreText></Link>
        );
      }

      if (this.props.onClick) {
        value = (
          <CoreInteractive onClick={this.props.onClick}><CoreText type={TextType.Span} color={Color.Link} fontSize={FontSize.Size4}>{this.props.value}</CoreText></CoreInteractive>
        );
      }

      return (
        <Layout padding={{ bottom: 1 }}>
          {warning}
          <CoreText type={TextType.Strong} color={color} fontSize={FontSize.Size4}>{this.props.name}:</CoreText>&nbsp;
          {value}&nbsp;
          {copy}
          <br />
        </Layout>
      );
    }
    return (<></>);
  }
}
