import { format } from 'date-fns';
import * as React from 'react';
import { TimeOption } from 'common/models/time-option';
import { ClickOutDetector } from 'components/click-out-detector';
import { TimeInteractable } from 'components/time-interactable';
import { Balloon, Input, InputType, Layout, Position } from 'twitch-core-ui';
import './styles.scss';

export enum RelativeTime {
  Now = 'NOW',
}

interface MinimumDate {
  getTime(): number;
  getTimezoneOffset(): number;
}

export function toUTC(now: MinimumDate): number {
  const localTime = now.getTime();
  const localOffset = now.getTimezoneOffset() * 60000;
  return localTime + localOffset;
}

export function toUTCDate(now: MinimumDate): Date {
  return new Date(toUTC(now));
}

export function isValid(time: Time) {
  if (
    time.hours > -1 &&
    time.hours < 24 &&
    time.minutes > -1 &&
    time.minutes < 60 &&
    (!time.seconds || (time.seconds > -1 && time.seconds < 60))
  ) {
    return true;
  }

  return false;
}

export function formatTime(time: Time, dateFormat: string): string {
  const date = new Date();
  date.setHours(time.hours);
  date.setMinutes(time.minutes);
  date.setSeconds(0);
  return formatDate(date, dateFormat);
}

export function formatDate(date: Date, dateFormat: string): string {
  return format(date, dateFormat);
}

export function parseTimeStringToTime(timeStr: string): Time {
  const parts = timeStr.split(':');
  if (parts.length !== 2) {
    throw new Error('invalid time string: ' + timeStr);
  }
  let hours = parseInt(parts[0], 10);
  const minutes = parseInt(parts[1].substring(0, 2), 10);
  const meridium = parts[1].substring(parts[1].length - 2);
  if (hours === 12 && meridium === 'am') {
    hours = 0;
  }
  if (meridium === 'pm' && hours < 12) {
    hours += 12;
  }
  if (
    isNaN(hours) ||
    hours < 0 ||
    hours > 23 ||
    isNaN(minutes) ||
    minutes < 0 ||
    minutes > 59
  ) {
    throw new Error('invalid time string: ' + timeStr);
  }
  return {
    hours,
    minutes,
  };
}

export interface Props {
  onChange: (time: Time | null, value: string) => void;
  initialTime: Time | null;
  customTimeOptions?: TimeOption[];
}

export interface Time {
  hours: number;
  minutes: number;
  seconds?: number;
}

interface State {
  isOpen: boolean;
  time?: Time | null;
  value: string;
  displayValue: string;
}

export class TimePicker extends React.Component<Props, State> {
  private selectedTime: HTMLDivElement;
  private timeSelector: HTMLDivElement;

  constructor(props: Props) {
    super(props);
    const defaultValue = props.initialTime
      ? this.timeToTimeString(props.initialTime)
      : '';
    this.state = {
      isOpen: false,
      displayValue: defaultValue,
      value: defaultValue,
    };
  }

  public componentDidUpdate(_: Props, prevState: State) {
    if (this.state.isOpen && !prevState.isOpen) {
      if (this.selectedTime) {
        this.timeSelector.scrollTop = this.selectedTime.offsetTop;
      }
    }
  }

  public render() {
    const timeOptions = this.props.customTimeOptions
      ? this.props.customTimeOptions.concat(this.generateTimeOptions())
      : this.generateTimeOptions();
    const options = timeOptions.map((option: TimeOption) => (
      <TimeInteractable
        {...option}
        key={option.value}
        onClick={this.onClickTime}
        selected={option.value === this.state.value}
        refDelegate={
          option.value === this.state.value
            ? this.setSelectedTimeRef
            : undefined
        }
      />
    ));
    return (
      <Layout
        position={Position.Relative}
        data-a-target="time-pick-field"
        className="time-picker"
      >
        <ClickOutDetector onClickOut={this.closeDropdown}>
          <Input
            type={InputType.Text}
            onFocus={this.onFocus}
            onChange={this.onInputChange}
            onBlur={this.onBlur}
            onKeyDown={this.onKeyDown}
            value={this.state.displayValue}
          />
          <Layout className="time-picker__balloon" position={Position.Absolute}>
            <Balloon
              noTail
              show={this.state.isOpen}
              data-a-target="time-selector-balloon"
              offsetX="0.1rem"
              offsetY="0.1rem"
            >
              <div
                className="time-picker__dropdown"
                ref={(ref: HTMLDivElement) => (this.timeSelector = ref)}
              >
                {options}
              </div>
            </Balloon>
          </Layout>
        </ClickOutDetector>
      </Layout>
    );
  }

  public generateTimeStrings(startDate: Date | undefined = undefined) {
    const times: string[] = [];
    const minutes = ['00', '30'];
    const startHour = startDate ? startDate.getHours() : 0;

    for (let hour = startHour; hour < 24; ++hour) {
      for (const minute of minutes) {
        if (hour > 11) {
          times.push(`${hour === 12 ? 12 : hour - 12}:${minute}pm`);
        } else {
          times.push(`${hour === 0 ? 12 : hour}:${minute}am`);
        }
      }
    }

    if (startDate && startDate.getMinutes() >= 30) {
      times.splice(0, 2);
    } else if (startDate) {
      times.splice(0, 1);
    }
    return times;
  }

  public generateTimeOptions(): TimeOption[] {
    const validTimeStrings = this.generateTimeStrings();
    const options: TimeOption[] = validTimeStrings.map(
      (timeString: string) => ({
        value: timeString,
        disabled: false,
      }),
    );
    return options;
  }

  public validateTime = (time: string): boolean => {
    const timeRegEx = /^([0-9]|0[0-9]|1[0-9]|2[0-3]):[0-5][0-9]\s*(am|pm)?$/i;
    return Boolean(time.match(timeRegEx));
  }

  public setValue(value: string, displayValue?: string, close?: boolean) {
    // See if the input matches any of our options, to set appropriate values
    // where they differ from displayed values.
    const timeOption = this.props.customTimeOptions
      ? this.props.customTimeOptions.find(
        (option: TimeOption) =>
          option.displayValue === (displayValue || value),
      )
      : null;
    value = timeOption && timeOption.value ? timeOption.value : value;
    displayValue =
      timeOption && timeOption.displayValue
        ? timeOption.displayValue
        : displayValue || value;
    this.setState({
      isOpen: this.state.isOpen && close ? false : this.state.isOpen,
      displayValue: displayValue ? displayValue : '',
      value,
      time: this.getTime(value),
    });
    this.notifyChange(value);
  }

  private getTime(value?: string) {
    try {
      return parseTimeStringToTime(value || this.state.value);
    } catch (e) {
      return null;
    }
  }

  private onFocus = () => {
    this.setState({
      isOpen: true,
    });
  }

  private onBlur = (e: React.FormEvent<HTMLInputElement>) => {
    const displayValue = this.formatTime(e.currentTarget.value);
    this.setValue(displayValue, displayValue);
  }

  // On click, the Balloon will close and the input will blur, triggering
  // onBlur, which is why we update the input value rather than the state.
  private onClickTime = (e: React.MouseEvent<HTMLLIElement>) => {
    const value = e.currentTarget.getAttribute('data-value') as string;
    const displayValue =
      (e.currentTarget.getAttribute('data-display-value') as string) || value;
    this.setValue(displayValue, displayValue, true);
  }

  private setSelectedTimeRef = (e: HTMLDivElement) => (this.selectedTime = e);

  private notifyChange(value: string) {
    if (this.props.onChange) {
      try {
        const newTime = parseTimeStringToTime(value);
        this.props.onChange(newTime, value);
      } catch (e) {
        this.props.onChange(null, value);
      }
    }
  }

  private onInputChange = (e: React.FormEvent<HTMLInputElement>) => {
    const displayValue = e.currentTarget.value;
    if (displayValue === this.state.displayValue) {
      return;
    }
    this.setValue(displayValue);
  }

  private onKeyDown = (e: React.KeyboardEvent<HTMLElement>) => {
    const key = e.charCode || e.keyCode;
    if (key === 9) { // Tab
      const displayValue = this.formatTime(this.state.displayValue);
      this.setValue(displayValue, displayValue, true);
    }
  }

  private closeDropdown = () => {
    this.setState({
      isOpen: false,
    });
  }

  private timeToTimeString(time: Time) {
    return formatTime(time, 'HH:mm');
  }

  private formatTime(timeValue: string) {
    let formattedTimeString = this.formatShortHand(timeValue);
    if (this.validateTime && this.validateTime(formattedTimeString)) {
      const tempDate = parseTimeStringToTime(formattedTimeString);
      formattedTimeString = this.timeToTimeString(tempDate);
    }
    return formattedTimeString;
  }

  private formatShortHand = (time: string): string => {
    const shortTimeRegEx = /^([0-9]|0[0-9]|1[0-9]|2[0-3])(am|pm)$/i;
    if (time.match(shortTimeRegEx)) {
      const meridiem = time.substr(time.length - 2);
      return time.replace(meridiem, `:00${meridiem}`);
    }
    return time;
  }
}
