import { APIResponse } from './apiresponse';

export const CONTENT_TYPE = 'Content-Type';
export const CONTENT_TYPE_JSON = 'application/json; charset=UTF-8';
export const CONTENT_TYPE_JSON_PREFIX = 'application/json';
export const ACCESS_TOKEN_COOKIE_NAME = 'amzn_sso_token';

export interface TimeStamp {
  seconds: number;
  nanos: number;
}

export interface Duration {
  seconds: number;
}

interface Headers {
  [id: string]: string;
}

/**
 * Error format returned for all Visage errors. See:
 * https://git-aws.internal.justin.tv/edge/visage/blob/master/internal/encoder/response_writer.go#L152-L158.
 */
export interface APIError {
  error: string;
  status: number;
  message: string;
  code: string;
}

/**
 * This interface acts as a modification of the RequestInit type:
 *
 * - headers: stricter so that we can merge default headers in with custom defined ones here.
 * - body: RequestInit doesn't allow a bespoke Object passed to fetch anymore, but we want our API to take an object.
 *   This can be a FormData, a pre-serialized string, or an Object (which will get serialized into a JSON string).
 *
 * TODO: replace Omit<> once TS 2.8 hits with the appropriate exclusion type.
 */
export interface Options {
  headers?: Headers;
  body?: FormData | string | Object;
  method?: string;
}

export interface GETOptions extends Options {
  method?: 'GET';
}

export interface PUTOptions extends Options {
  method?: 'PUT';
}

export interface POSTOptions extends Options {
  method?: 'POST';
}

export interface DELETEOptions extends Options {
  method?: 'DELETE';
}

export class API {
  public static csrfToken: string;
  /**
   * GET:
   *
   * This request will not reject its promise when a response comes back with a non-2xx status,
   * or with a malformed request object. Use response.error or response.requestError after resolving
   * instead to capture errors.
   */
  public static async get<T>(
    path: string,
    options: GETOptions = {},
  ): Promise<APIResponse<T>> {
    return await this.request<T>(path, {
      ...options,
      method: 'GET',
    });
  }

  public static async put<T>(
    path: string,
    options: PUTOptions = {},
  ): Promise<APIResponse<T>> {
    return await this.request<T>(path, {
      ...options,
      method: 'PUT',
    });
  }

  public static async post<T>(
    path: string,
    options: POSTOptions = {},
  ): Promise<APIResponse<T>> {
    return await this.request<T>(path, {
      ...options,
      method: 'POST',
    });
  }

  public static async delete<T>(
    path: string,
    options: DELETEOptions = {},
  ): Promise<APIResponse<T>> {
    return await this.request<T>(path, {
      ...options,
      method: 'DELETE',
    });
  }

  public static async request<T>(
    path: string,
    options: Options = {},
  ): Promise<APIResponse<T>> {
    options = this.constructOptions(options, this.csrfToken);
    const contentType = options.headers
      ? options.headers[CONTENT_TYPE]
      : undefined;

    // Transform Options.body to something fetch can handle
    const body = this.serialize(options.body, contentType);
    const fetchOptions: RequestInit = { ...options, body };

    let res: Response = await this._fetch(path, fetchOptions);

    let csrf = res.headers.get('x-csrf-token');
    if (csrf) {
      this.csrfToken = csrf;
    }

    if (res.status == 401) {
      window.location.reload() // Sucks but retrivea broke our old logic and this is what we have until we can migrate off
    }

    return await this.constructAPIResponse<T>(res);
  }

  /**
   * Returns a URL who's "host" parameter is determined by Twilight configuration options.
   *
   * @param {string} path Example: /users/17272570
   * @returns {URL}
   */
  // TODO: should be based on config
  public static getAPIURL(path: string): URL {
    const baseUrl = process.env.REACT_APP_API_URL || window.location.href;
    return new URL(path, baseUrl);
  }

  private static postMessageListenerAdded: boolean | undefined = false;

  private static reauthPromise: Promise<boolean> | undefined;

  private static constructOptions<T extends Options>(options: T, csrfToken: string | undefined): T {
    // TypeScript does not allow spreading a generic, hence the usage of `assign`.
    const defaultHeaders = {
      'Content-Type': CONTENT_TYPE_JSON,
    };

    if (csrfToken) {
      defaultHeaders['x-csrf-token'] = csrfToken;
    }

    options = Object.assign({}, options, {
      headers: {
        ...defaultHeaders,
        ...options.headers,
      },
    });

    return options;
  }

  private static reauthPromiseResolve?(_?: boolean | PromiseLike<boolean>): any;

  private static handlePostMessage(event: MessageEvent) {
    // Same origin only
    if (event.origin !== window.origin) {
      return;
    }

    if (event.data['type'] == 'midway_reauth' && event.data['result'] == 'success' && this.reauthPromise && this.reauthPromiseResolve) {
      this.reauthPromiseResolve(true);
      this.reauthPromise = undefined;
      this.reauthPromiseResolve = undefined;
    }
  }

  private static async constructAPIResponse<T extends {}>(
    res: Response,
  ): Promise<APIResponse<T>> {
    const apiResponse = new APIResponse<T>();

    apiResponse.status = res.status;

    try {
      const body = await res.json();

      if (res.ok) {
        apiResponse.body = body as T;
      } else {
        apiResponse.error = body as APIError;
      }
    } catch (err) {
      // failure to parse json should only be an error if content type is also json
      // At this point there are already many tests that don't mock headers
      // therefore this code ensures headers exist before verifying content type
      // and checking content type only on failures
      apiResponse.error = {
        status: 500,
        error: 'Unknown Error',
        code: 'UNKNOWN',
        message: 'Something went wrong and my code doesn\'t know what :(',
      };

      if (res.headers && res.headers.get) {
        const contentType = res.headers.get(CONTENT_TYPE);
        if (
          contentType &&
          contentType.indexOf(CONTENT_TYPE_JSON_PREFIX) !== -1
        ) {
          apiResponse.requestError = err;

        }
      }
    }

    return apiResponse;
  }

  private static async _fetch(
    path: string,
    options: RequestInit = {},
  ): Promise<Response> {
    return await fetch(this.getAPIURL(path).toString(), options);
  }

  /**
   * This transforms an object into one that fetch's RequestInit can handle
   * as a body field. This limits the serialized output to null, FormData,
   * a pre-serialized string, or a JSON string.
   */
  private static serialize(
    body?: FormData | Object,
    contentType?: string,
  ): FormData | string | null {
    if (contentType === CONTENT_TYPE_JSON) {
      return JSON.stringify(body);
    } else if (typeof body === 'string') {
      return body;
    } else if (body && FormData.prototype.isPrototypeOf(body)) {
      return body as FormData;
    } else if (!body) {
      return null;
    } else {
      console.log(
        'Could not serialize this request body for the content-type provided.',
      );
      console.log(
        'attempting to serialize object with a non-JSON content-type',
      );
      console.log(contentType);
      return null;
    }
  }
}
