import * as React from 'react';
import { Link } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { APIError } from 'lib/api';
import { CoreText, Layout, LoadingSpinner, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { getStreamConfigs, StreamConfig } from '../../api';
import { Paging, Position } from 'components/paging'

interface Props {
  customer_id: string;
  region: string;
}

interface State {
  stream_configs?: StreamConfig[];
  processing: boolean;
  next_page_token?: string;
  page_stack: string[];
}

export class ChannelList extends React.Component<Props, State> {
  public state: State = {
    processing: true,
    page_stack: [],
  };

  public render() {
    let content = <></>

    if (this.state && this.state.stream_configs) {
      content = (
        <TableBody>
          {this.state.stream_configs.map((streamConfig, index) => (
            <TableRow key={index}>
              <TableCell>{streamConfig.stream_config_id}</TableCell>
              <TableCell>{streamConfig.nickname}</TableCell>
              <TableCell>{(!!streamConfig.playback_auth_required).toString()}</TableCell>
              <TableCell><Link to={'/channelviewer/' + this.props.customer_id + '/' + this.props.region + '/' + streamConfig.stream_config_id}>Details</Link></TableCell>
            </TableRow>
          ))}
        </TableBody>
      );
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 3 }}>
          <CoreText type={TextType.H3}>Channels</CoreText>
        </Layout>
        <Layout>
        <Paging loading={this.state.processing} position={Position.Before} prevPageExists={this.state.page_stack.length > 0} nextPageExists={!!this.state.next_page_token} triggerNextPage={() => this.nextPage()} triggerPrevPage={() => this.prevPage()}>
          <Table>
            <TableHeader>
              <TableHeading label="Stream Config ID" />
              <TableHeading label="Nickname" />
              <TableHeading label="Playback Auth Required" />
              <TableHeading label="Details" />
            </TableHeader>
            {content}
          </Table>
        </Paging>
        </Layout>
      </Layout>
    );
  }

  public componentWillMount() {
    this.getStreamConfigs('');
  }

  private nextPage() {
    if (!this.state) {
      return
    }

    let ps = this.state.page_stack;
    ps.push(this.state.next_page_token)

    this.setState({
      page_stack: ps,
    })

    this.getStreamConfigs(this.state.next_page_token);
  }

  private prevPage() {
    let ps = this.state.page_stack;
    ps.pop()

    this.setState({
      page_stack: ps,
    })

    let page_token = '';

    if (ps.length > 0){
      page_token = ps[ps.length - 1]
    }

    this.getStreamConfigs(page_token);
  }

  private async getStreamConfigs(page_token :string) {
    this.setState({
      processing: true,
    });

    await getStreamConfigs(this.props.customer_id, this.props.region, page_token).then((streamConfigResponse) => {
      this.setState({
        stream_configs: streamConfigResponse.stream_configs,
        next_page_token: streamConfigResponse.next_page_token,
        processing: false,
      });
    }, (reason: APIError) => {
      this.setState({
        processing: false,
      });

      if (reason.status == 404){
        return
      }

      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Getting Stream Config List',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

ChannelList.contextType = AppContext;
