import * as React from 'react';
import { match, Redirect } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { DataClassification, PanelItem } from 'components/panel-item/panel-item';
import { APIError } from 'lib/api';
import { Button, CoreText, Display, FlexDirection, FlexWrap, FontSize, Layout, LoadingSpinner, StyledLayout, TextType } from 'twitch-core-ui';
import { getStreamConfigDetail, StreamConfigDetail } from '../../api';

interface URLParams {
  customer_id: string;
  region: string;
  stream_config_id: string;
}

interface Props {
  match: match<URLParams>;
}

interface State {
  streamConfig?: StreamConfigDetail;
  navigate?: string;
  processing: boolean;
}

export class ChannelDetail extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    streamConfig: undefined,
    processing: true,
  };

  public fetchPromise: Promise<StreamConfigDetail | undefined> | undefined;

  public render() {
    if (this.state.navigate !== undefined) {
      // Since we're often navigating to the same component
      this.setState({
        navigate: undefined,
      });
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.processing) {
      return (
        <LoadingSpinner />
      );
    }

    if (this.state.streamConfig) {
      let channelName = this.state.streamConfig.stream_config.stream_config_id;
      if (this.state.streamConfig.stream_config.nickname) {
        channelName = this.state.streamConfig.stream_config.nickname;
      }

      let streamKeys = <></>

      if (this.state.streamConfig.stream_keys) {
        streamKeys = (
          <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
            <Layout padding={{ bottom: 2 }}>
              <CoreText type={TextType.H3}>Stream Keys</CoreText>
            </Layout>
            <Layout padding={{ bottom: 2 }}>
              {this.state.streamConfig.stream_keys.map((skey, index) => (
                <Layout key={index}>
                  <PanelItem dataClassification={DataClassification.CustomerOnly} name="Stream Key ID" value={skey.id} />
                </Layout>
              ))}
            </Layout>
          </StyledLayout>
        )
      }

      return (
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row} padding={{ bottom: 5 }}>
            <Layout flexGrow={1}>
              <CoreText type={TextType.H2}>Channel Details for {channelName}</CoreText>
            </Layout>
            <Layout>
              <Layout margin={{ left: 1 }} display={Display.InlineBlock}>
                <Button linkTo={'/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.stream_config_id}>View Streaming Sessions</Button>
              </Layout>
              <Layout margin={{ left: 1 }} display={Display.InlineBlock}>
                <Button linkTo={'/account/' + this.props.match.params.customer_id + '/' + this.props.match.params.region}>Back to Account View</Button>
              </Layout>
            </Layout>
          </Layout>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row} flexWrap={FlexWrap.Wrap}>
            <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
              <Layout padding={{ bottom: 2 }}>
                <CoreText type={TextType.H3}>Channel Info</CoreText>
              </Layout>
              <Layout padding={{ bottom: 2 }}>
                <PanelItem name="Channel ID" dataClassification={DataClassification.CustomerOnly} value={this.state.streamConfig.stream_config.stream_config_id} />
                <PanelItem name="Channel Name" dataClassification={DataClassification.CustomerOnly} value={this.state.streamConfig.stream_config.nickname} />
                <PanelItem name="Pricing Mode" dataClassification={DataClassification.CustomerOnly} value={this.state.streamConfig.stream_config.pricing_mode} />
                <PanelItem name="Latency Mode" dataClassification={DataClassification.CustomerOnly} value={this.state.streamConfig.stream_config.playback_latency_mode} />
                <PanelItem name="Recording Bucket" dataClassification={DataClassification.CustomerOnly} value={this.state.streamConfig.stream_config.recording_s3_bucket_name} />
              </Layout>
            </StyledLayout>
          {streamKeys}
          </Layout>
        </Layout>
      );
    }

    return (
      <div>We didn't get the channel for some reason</div>
    );
  }

  public componentDidMount() {
    this.getStreamConfigDetail();
  }

  public componentDidUpdate(prevProps: Props) {
    if (this.props.match.params.stream_config_id === prevProps.match.params.stream_config_id) {
      return;
    }

    this.setState({
      processing: true,
    });

    this.getStreamConfigDetail();
  }

  private async getStreamConfigDetail() {
    if (this.fetchPromise) {
      Promise.reject(this.fetchPromise);
      this.fetchPromise = undefined;
    }

    this.fetchPromise = getStreamConfigDetail(this.props.match.params.customer_id, this.props.match.params.region, this.props.match.params.stream_config_id);

    await this.fetchPromise.then((streamConfig) => {
      this.fetchPromise = undefined;
      this.setState({
        streamConfig,
        processing: false,
      });
    }, (reason: APIError) => {
      this.context.showModal(
        (
          <CoreText fontSize={FontSize.Size4}>{reason.message}</CoreText>
        ),
        {
          title: 'Error Getting Channel Detail',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

ChannelDetail.contextType = AppContext;
