import * as React from 'react';
import {Redirect} from 'react-router-dom';
import {AppContext} from 'common/appcontext';
// @ts-ignore
import {DataClassification, PanelItem, PanelItemWarning} from 'components/panel-item/panel-item';
import {APIError} from 'lib/api';
// @ts-ignore
import {
  AlignContent,
  AlignItems,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FormGroupOrientation,
  JustifyContent,
  Layout,
  LoadingSpinner,
  StyledLayout,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  TextType
} from 'twitch-core-ui';
import {getPoPHealth, PoPHealth} from '../../api';
import {PoPHealthItem} from '../pophealth';

import './styles.css';

export interface State {
  navigate: string | undefined;
  loading: boolean;
  pops?: PoPHealth[];
}

export class PoPList extends React.Component<{}, State> {
  public state: State = {
    navigate: undefined,
    loading: true,
  };

  private reloadTimeout: NodeJS.Timer;
  private healthPromiseRejector: (reason?: any) => void;

  public render() {
    if (this.state.loading) {
      return (
          <Layout>
            <LoadingSpinner />
          </Layout>
      );
    }

    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    let list = <></>;

    if (this.state && this.state.pops && this.state.pops.length > 0) {
      list = (
          <TableBody>
            {this.state.pops.map((pop, index) => (
                <PoPHealthItem key={index} pop={pop} />
            ))}
          </TableBody>
      );
    }

    return (
        <Layout textAlign={TextAlign.Left}>
          <Table alternateRows={true}>
            <TableHeader>
              <TableHeading label="Name" />
              <TableHeading label="Flags" />
              <TableHeading label="MW" />
              <TableHeading label="BE/MW" />
              <TableHeading label="CCB & PTS" />
              <TableHeading label="Backbone" />
              <TableHeading label="Edge" />
            </TableHeader>
            {list}
          </Table>
        </Layout>
    );
  }

  public componentDidMount() {
    this.getPoPHealth();
  }

  public componentWillUnmount() {
    if (this.reloadTimeout) {
      clearTimeout(this.reloadTimeout);
    }

    if (this.healthPromiseRejector) {
      this.healthPromiseRejector({code: 'UNMOUNT'})
      this.healthPromiseRejector = null;
    }
  }

  private getPoPHealth = () => {
    this.reloadTimeout = null;

    let popHealthPromise = new Promise<PoPHealth[]>((resolve, reject) => {
      this.healthPromiseRejector = reject;
      return getPoPHealth().then(resolve, reject)
    });

    popHealthPromise.then((pops: PoPHealth[]) => {
      this.setState({
        loading: false,
        pops,
      });

      this.healthPromiseRejector = null;
      this.reloadTimeout = setTimeout(this.getPoPHealth, 10000);
    }, (reason: APIError) => {
      if (reason.code == 'UNMOUNT') {
        return
      }

      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Retrieving PoP List',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

PoPList.contextType = AppContext;
