import {
  endOfDay,
  startOfDay,
  subDays,
} from 'date-fns';
import * as React from 'react';
import { Link, Redirect } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { DateRangePicker } from 'components/daterangepicker';
import { PresetID, PresetInterval } from 'components/daterangepicker-presets';
import { Option, SelectInput } from 'components/select-input';
import { TextInput } from 'components/text-input';
import { APIError } from 'lib/api';
// @ts-ignore
import { Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { listQueries, QueryStatus, QuerySummary, submitQuery } from '../../api';

export interface State {
  navigate: string | undefined;
  loading: boolean;
  // For new reports
  reportType: string | undefined;
  dateRange: PresetInterval;
  playSessionID: string | undefined;
  broadcastSessionID: string | undefined;

  // For report list
  queries: QuerySummary[];
}

// TODO, make this shit more modular like the view stuff
export class ReportsHome extends React.Component<{}, State> {
  public static ReportTypes: Option[] = [
    {
      label: 'Select Report Type',
      value: 'new_report',
      selected: true,
    },
    {
      label: 'Play Session Summary',
      value: 'play_session_info',
      selected: false,
    },
    {
      label: 'Broadcast Session Summary',
      value: 'broadcast_session_info',
      selected: false,
    },
  ];

  public state: State = {
    navigate: undefined,
    loading: false,
    reportType: undefined,
    dateRange: {
      // Because why write this in a way that I can actually retrieve a preset interval programatically....
      start: startOfDay(subDays(new Date(), 6)),
      end: endOfDay(new Date()),
      id: PresetID.Last7Days,
    },
    playSessionID: undefined,
    broadcastSessionID: undefined,
    queries: [],
  };

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    let list = <></>;

    if (this.state && this.state.queries && this.state.queries.length > 0) {
      list = (
        <Layout>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H2}>Recent Reports</CoreText>
          </Layout>
          <Table>
            <TableHeader>
              <TableHeading label="Query" />
              <TableHeading label="Requested At" />
              <TableHeading label="Status" />
              <TableHeading label="Details" />
            </TableHeader>
            <TableBody>
              {this.state.queries.map((query, index) => (
                <TableRow key={index}>
                  <TableCell>{query.label}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(query.requested_at)}</TableCell>
                  <TableCell>{QueryStatus[query.status ? query.status : 0]}</TableCell>
                  <TableCell><Link to={'/reports/' + query.query_id}>Details</Link></TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </Layout>
      );
    }

    let dateRange = (
      <Layout padding={{ top: 2, right: 2, bottom: 1, left: 2 }} justifyContent={JustifyContent.Center}>
        <DateRangePicker
          interval={this.state.dateRange}
          onCommit={this.onDateSelect}
        />
      </Layout>
    );

    let buttons = (
      <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={{ bottom: 3 }}>
        <Button disabled={!this.isValid(this.state)} onClick={this.submitReport} type={ButtonType.Primary}>Submit</Button>
      </Layout>
    );

    let inputs = (<></>);
    switch (this.state.reportType) {
      case 'play_session_info':
        inputs = this.playSessionSummaryInputs();
        break;
      case 'broadcast_session_info':
        inputs = this.broadcastSessionSummaryInputs();
        break;
      default:
        dateRange = (<></>);
        buttons = (<></>);
        break;
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 2 }} display={Display.Flex}>
          <Layout flexGrow={1}>
            <CoreText type={TextType.H2}>Reports</CoreText>
          </Layout>
        </Layout>
        <Layout padding={{ bottom: 2 }} display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout padding={{ bottom: 3 }}>
            <SelectInput label="New Report" setValue={this.setReportType} defaultValue={undefined} orientation={FormGroupOrientation.Horizontal} options={ReportsHome.ReportTypes} />
          </Layout>
        </Layout>
        {dateRange}
        {inputs}
        {buttons}
        {list}
      </Layout>
    );
  }

  public componentDidMount() {
    document.addEventListener('keydown', this.onKeyDown);

    this.listQueries();
  }

  public componentWillUnmount() {
    document.removeEventListener('keydown', this.onKeyDown);
  }

  private playSessionSummaryInputs() {
    return (
      <Layout padding={{ bottom: 3 }}>
        <TextInput label="Play Session ID" setValue={this.setPlaySessionID} orientation={FormGroupOrientation.Horizontal} />
      </Layout>
    );
  }

  private broadcastSessionSummaryInputs() {
    return (
      <Layout padding={{ bottom: 3 }}>
        <TextInput label="Ingest Session ID" setValue={this.setBroadcastSessionID} orientation={FormGroupOrientation.Horizontal} />
      </Layout>
    );
  }

  private onKeyDown = (ev: KeyboardEvent) => {
    if (ev.keyCode == 13 && this.isValid(this.state)) {
      this.submitReport();
    }
  }

  private isValid(state: State): boolean {
    if (!state) {
      return false;
    }

    switch (state.reportType) {
      case 'new_report':
        return false;
      case 'play_session_info':
        if (!state.playSessionID) {
          return false;
        }
        return true;
      case 'broadcast_session_info':
        if (!state.broadcastSessionID) {
          return false;
        }
        return true;
      default:
        return false;
    }
  }

  private onDateSelect = (interval: PresetInterval) => {
    this.setState({
      dateRange: interval,
    });
  }

  private setPlaySessionID = (playSessionID: string | undefined): string | undefined => {
    this.setState({
      playSessionID,
    });

    if (!playSessionID) {
      return 'Please supply a play session ID';
    }

    return undefined;
  }

  private setBroadcastSessionID = (broadcastSessionID: string | undefined): string | undefined => {
    this.setState({
      broadcastSessionID,
    });

    if (!broadcastSessionID) {
      return 'Please supply an ingest session ID';
    }

    return undefined;
  }

  private setReportType = (reportType: string | undefined): string | undefined => {
    this.setState({
      reportType,
    });

    if (!reportType) {
      return 'Please supply a report Type';
    }

    return undefined;
  }

  private listQueries = () => {
    listQueries().then((queries) => {
      this.setState({ queries });
    });
  }

  private submitReport = () => {
    if (!this.isValid(this.state)) {
      return;
    }

    submitQuery({
      // @ts-ignore - reportType can't be undefined if state is valid
      query_type: this.state.reportType,
      start: this.state.dateRange.start,
      end: this.state.dateRange.end,
      play_session_id: this.state.playSessionID,
      broadcast_session_id: this.state.broadcastSessionID,
    }).then((queryID: string) => {
      this.setState({
        navigate: '/reports/' + queryID,
      });
    }, (reason: APIError) => {
      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Generating Report',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

ReportsHome.contextType = AppContext;
